package org.lsst.ccs.subsystem.comcamvacuum.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.border.TitledBorder;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.comcamvacuum.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.comcamvacuum.constants.ConditionState;
import org.lsst.ccs.subsystem.comcamvacuum.constants.DeviceState;
import org.lsst.ccs.subsystem.comcamvacuum.constants.SwitchEnable;
import org.lsst.ccs.subsystem.comcamvacuum.constants.SwitchState;
import org.lsst.ccs.subsystem.comcamvacuum.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class ComCamVacControlPanel extends javax.swing.JPanel {

    static final Font FONT = new java.awt.Font("Tahoma", 1, 12);
    static final Color RED = new Color(175, 0, 0), GREEN = new Color(0, 175, 0),
                       BLUE = new Color(0, 75, 175), PURPLE = new Color(175, 0, 175);
    static final int SWTP_PUMP = 0, SWTP_VALVE = 1, SWTP_KOOLANCE = 2, SWTP_CRYOTEL = 3, SWTP_CRYOTEL_POWER = 4;
    static final String[] switchNames = new String[VacSysState.NUM_SWITCHES];
    static {
        switchNames[VacSysState.SW_CRYO_TURBO] = "Cryo Turbo Pump";
        switchNames[VacSysState.SW_CRYO_ION_PUMP1] = "Cryo Ion Pump 1";
//        switchNames[VacSysState.SW_CRYO_ION_PUMP2] = "Cryo Ion Pump 2";
//        switchNames[VacSysState.SW_CRYO_ION_PUMP3] = "Cryo Ion Pump 3";
//        switchNames[VacSysState.SW_CRYO_ION_PUMP4] = "Cryo Ion Pump 4";
//        switchNames[VacSysState.SW_CRYO_ION_PUMP5] = "Cryo Ion Pump 5";
//        switchNames[VacSysState.SW_CRYO_ION_PUMP6] = "Cryo Ion Pump 6";
        switchNames[VacSysState.SW_CRYO_VALVE] = "VAT Valve";
        switchNames[VacSysState.SW_HX_TURBO] = "Hx Turbo Pump";
        switchNames[VacSysState.SW_HX_ION_PUMP1] = "Hx Ion Pump 1";
//        switchNames[VacSysState.SW_HX_ION_PUMP2] = "Hx Ion Pump 2";
        switchNames[VacSysState.SW_HX_VALVE] = "Hx Gate Valve";
        switchNames[VacSysState.SW_OR_ION_PUMP] = "O-ring Ion Pump";
        switchNames[VacSysState.SW_OR_FPP_VALVE] = "O-ring Fpp Valve";
        switchNames[VacSysState.SW_OR_FH_VALVE] = "O-ring Fh Valve";
        switchNames[VacSysState.SW_OR_L3H_VALVE] = "O-ring L3h Valve";
        switchNames[VacSysState.SW_OR_L3_VALVE] = "O-ring L3 Valve";
        switchNames[VacSysState.SW_KOOLANCE] = "KOOLANCE";
        switchNames[VacSysState.SW_CRYOTEL_CRYO] = "CryoTel Cryo";
        switchNames[VacSysState.SW_CRYOTEL_COLD1] = "CryoTel Cold1";
        switchNames[VacSysState.SW_CRYOTEL_COLD2] = "CryoTel Cold2";
        switchNames[VacSysState.SW_CRYOTEL_PWR] = "CryoTel POWER";
    }
    static final int[] switchTypes = new int[VacSysState.NUM_SWITCHES];
    static {
        switchTypes[VacSysState.SW_CRYO_TURBO] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_ION_PUMP1] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_CRYO_ION_PUMP2] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_CRYO_ION_PUMP3] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_CRYO_ION_PUMP4] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_CRYO_ION_PUMP5] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_CRYO_ION_PUMP6] = SWTP_PUMP;
        switchTypes[VacSysState.SW_CRYO_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_HX_TURBO] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_ION_PUMP1] = SWTP_PUMP;
//        switchTypes[VacSysState.SW_HX_ION_PUMP2] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_ION_PUMP] = SWTP_PUMP;
        switchTypes[VacSysState.SW_OR_FPP_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_FH_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3H_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_OR_L3_VALVE] = SWTP_VALVE;
        switchTypes[VacSysState.SW_KOOLANCE] = SWTP_KOOLANCE;
        switchTypes[VacSysState.SW_CRYOTEL_CRYO] = SWTP_CRYOTEL;
        switchTypes[VacSysState.SW_CRYOTEL_COLD1] = SWTP_CRYOTEL;
        switchTypes[VacSysState.SW_CRYOTEL_COLD2] = SWTP_CRYOTEL;
        switchTypes[VacSysState.SW_CRYOTEL_PWR] = SWTP_CRYOTEL_POWER;
    }
    static final String[] conditionNames = new String[VacSysState.NUM_CONDITIONS];
    static {
        conditionNames[VacSysState.COND_CR_GATE_AO] = "VAT Valve Can't Open";
        conditionNames[VacSysState.COND_CR_GATE_NFC] = "VAT Valve Forced Shut";
        conditionNames[VacSysState.COND_CR_VACUUM] = "Cryo Vacuum Bad";
        conditionNames[VacSysState.COND_CR_PUMP] = "Cryo Turbo Pump Bad";
        conditionNames[VacSysState.COND_HX_GATE_AO] = "Hx Gate Can't Open";
        conditionNames[VacSysState.COND_HX_GATE_NFC] = "Hx Gate Forced Shut";
        conditionNames[VacSysState.COND_HX_VACUUM] = "Hx Vacuum Bad";
        conditionNames[VacSysState.COND_HX_PUMP] = "Hx Turbo Pump Bad";
    }

    final private CommandSender sender;
    private String subsys;
    private double mainPeriod = 5.0;
    private int switchMask, conditionMask;

    private final JPanel mainPanel = new JPanel();

    private final JPanel headPanel = new JPanel();
    private final JLabel stateLabel = new JLabel("System State:  ");
    private final JLabel stateStatus = new JLabel("STOPPED");

    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField();
    private final JLabel periodUnitsLabel = new JLabel("sec");

    //private final JPanel[] switchPanel = new JPanel[VacuumState.NUM_SWITCHES];
    private final JPanel switchPanel = new JPanel();
    private final JLabel[] switchLabel = new JLabel[VacSysState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[VacSysState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[VacSysState.NUM_SWITCHES];

    private final JPanel conditionPanel = new JPanel();
    private final JLabel[] conditionLabel = new JLabel[VacSysState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[VacSysState.NUM_CONDITIONS];
    private final JButton[] conditionResetBtn = new JButton[VacSysState.NUM_CONDITIONS];

    public ComCamVacControlPanel(CommandSender cons) {
        this.sender = cons;
        initComponents();
        (new UpdateVacStatus(new VacSysState())).run();
        (new DisableSystem()).run();
    }

    public void setSubsystem(String name) {
        this.subsys = name;
    }

    private void initComponents() {

        // System state
        stateLabel.setFont(FONT);
        stateStatus.setFont(FONT);
        Dimension d = stateStatus.getPreferredSize();
        d.width = 150;
        stateStatus.setMinimumSize(d);
        stateStatus.setPreferredSize(d);

        // Update period
        periodLabel.setFont(FONT);
        d = periodTextField.getPreferredSize();
        d.width = 45;
        periodTextField.setMinimumSize(d);
        periodTextField.setPreferredSize(d);
        periodTextField.setHorizontalAlignment(SwingConstants.CENTER);
        periodTextField.setText("...");
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });
        periodUnitsLabel.setFont(FONT);

        // Switch states
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            JLabel label = switchLabel[sw] = new JLabel(switchNames[sw]);
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 130;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            label = switchStatus[sw] = new JLabel("XXX");
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 80;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            int swType = switchTypes[sw];
            JRadioButton buttonOff = switchOffRB[sw] = new JRadioButton(swType == SWTP_PUMP ||
                    swType == SWTP_KOOLANCE || swType == SWTP_CRYOTEL || swType == SWTP_CRYOTEL_POWER ? "Off" : "Shut");
            buttonOff.setFont(FONT);
            //buttonOff.setText("Off");
            buttonOff.setFocusable(false);
            buttonOff.setName(Integer.toString(sw));
            buttonOff.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            JRadioButton buttonOn = switchOnRB[sw] = new JRadioButton(swType == SWTP_PUMP  || 
                    swType == SWTP_KOOLANCE || swType == SWTP_CRYOTEL || swType == SWTP_CRYOTEL_POWER ? "On" : "Open");
            buttonOn.setFont(FONT);
            //buttonOn.setText("On");
            buttonOn.setFocusable(false);
            buttonOn.setName(Integer.toString(sw));
            buttonOn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(buttonOff);
            bg.add(buttonOn);
        }

        // Condition states
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            JLabel label = conditionLabel[cond] = new JLabel(conditionNames[cond]);
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 170;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            label = conditionStatus[cond] = new JLabel("XXX");
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 80;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            JButton button = conditionResetBtn[cond] = new JButton("Reset");
            Insets i = button.getMargin();
            i.left -= 3;
            i.right -= 3;
            button.setMargin(i);
            d = button.getPreferredSize();
            d.height -= 9;
            //d.width -= 10;
            //button.setMinimumSize(d);
            button.setPreferredSize(d);
            button.setFont(FONT);
            button.setFocusable(false);
            button.setName(Integer.toString(cond));
            button.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    clearCondition(Integer.valueOf(name));
                }
            });
        }


        // Lay out all the sub-panels
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Add state & period items to head panel
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 0, 0, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.EAST;
        headPanel.add(stateLabel, gbh);
        gbh.gridx++;
        gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(stateStatus, gbh);
        gbh.gridx++;
        gbh.gridwidth = 2;
        gbh.anchor = GridBagConstraints.EAST;
        headPanel.add(periodLabel, gbh);
        gbh.gridx += 2;
        gbh.gridwidth = 1;
        gbh.anchor = GridBagConstraints.WEST;
        headPanel.add(periodPanel, gbh);
        gbh.gridx = 0;

        // Prepare the switch panel
        TitledBorder border = BorderFactory.createTitledBorder("Switches");
        border.setTitleJustification(TitledBorder.CENTER);
        switchPanel.setBorder(border);
        switchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.gridx = 0;
        gbs.gridy = 0;
        gbs.insets.bottom = -8;
        switchPanel.add(new JLabel(" "), gbs);

        // Prepare the conditions panel
        border = BorderFactory.createTitledBorder("PLC Error Conditions");
        border.setTitleJustification(TitledBorder.CENTER);
        conditionPanel.setBorder(border);
        conditionPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets.bottom = -8;
        conditionPanel.add(new JLabel(" "), gbc);

        // Lay out the main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(headPanel, gbm);
        gbm.gridy++;
        mainPanel.add(switchPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        mainPanel.add(conditionPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);

    }

    private void updateSwitchLayout(int switches) {
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switchMask & (1 << sw)) != 0) {
                switchPanel.remove(switchLabel[sw]);
                switchPanel.remove(switchStatus[sw]);
                switchPanel.remove(switchOffRB[sw]);
                switchPanel.remove(switchOnRB[sw]);
            }
        }
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        int colm = 0;
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switches & (1 << sw)) != 0) {
                if (colm == 0) {
                    gbs.gridx = 0;
                    gbs.gridy++;
                }
                gbs.insets.top = 0;
                gbs.insets.left = colm == 0 ? 4 : 50;
                //gbs.anchor = GridBagConstraints.NORTHEAST;
                switchPanel.add(switchLabel[sw], gbs);
                gbs.insets.left = 0;
                gbs.gridx++;
                //gbs.anchor = GridBagConstraints.NORTHWEST;
                switchPanel.add(switchStatus[sw], gbs);
                gbs.gridx++;
                gbs.insets.top = -4;
                switchPanel.add(switchOffRB[sw], gbs);
                gbs.gridx++;
                gbs.insets.left = 6;
                gbs.insets.right = 3;
                switchPanel.add(switchOnRB[sw], gbs);
                gbs.insets.right = 0;
                gbs.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        switchMask = switches;
    }

    private void updateConditionLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if ((conditionMask & (1 << cond)) != 0) {
                conditionPanel.remove(conditionLabel[cond]);
                conditionPanel.remove(conditionStatus[cond]);
                conditionPanel.remove(conditionResetBtn[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = 0;
                gbc.insets.bottom = 4;
                gbc.insets.left = colm == 0 ? 3 : 50;
                conditionPanel.add(conditionLabel[cond], gbc);
                gbc.insets.left = 0;
                gbc.gridx++;
                conditionPanel.add(conditionStatus[cond], gbc);
                gbc.gridx++;
                gbc.insets.top = -1;
                gbc.insets.right = 4;
                conditionPanel.add(conditionResetBtn[cond], gbc);
                gbc.insets.right = 0;
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        conditionMask = conditions;
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            sender.sendCommand(subsys, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(mainPeriod));
        }
    }

    private void setSwitchOn(int sw, boolean value) {
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(subsys, "setSwitchOn", sw, value);
    }
    
    private void clearCondition(int cond) {
        conditionResetBtn[cond].setEnabled(false);
        sender.sendCommand(subsys, "clearCondition", cond);
    }
    
    public void updateControlPanel(VacSysState rs) {
        SwingUtilities.invokeLater(new UpdateVacStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateVacStatus implements Runnable {

        private final VacSysState vs;

        UpdateVacStatus(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            stateStatus.setText("RUNNING");
            stateStatus.setForeground(GREEN);
            periodTextField.setEnabled(true);
            mainPeriod = vs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(mainPeriod));
            int switches = 0;
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                if (!vs.hasSwitch(sw)) continue;
                switches |= (1 << sw);
                SwitchState state = vs.getSwitchState(sw);

                String text;
                Color color;
                SwitchEnable enable = vs.getSwitchEnable(sw);
                if (enable == SwitchEnable.OFF || enable == SwitchEnable.WAS_OFF) {
                    text = "DISABLD";
                    color = RED;
                }
                else {
                    text = state.name();
                    if (switchTypes[sw] == SWTP_VALVE) {
                        text = state == SwitchState.OFF ? "SHUT" : state == SwitchState.ON ? "OPEN" : text;
                    }
                    color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? GREEN : BLUE;
                    DeviceState devState = vs.getDeviceState(sw);
                    if (devState != null && state != SwitchState.OFFLINE) {
                        text = devState.name();
                        if (devState == DeviceState.FAILED) {
                            color = RED;
                        }
                    }
                }
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            if (switches != switchMask) {
                updateSwitchLayout(switches);
            }
            int conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                if (!vs.hasCondition(cond)) continue;
                conditions |= (1 << cond);
                ConditionState state = vs.getCondition(cond);
                String text = state.name();
                Color color = state == ConditionState.OFFLINE ? BLUE :
                              state == ConditionState.CLEAR ? GREEN :
                              state == ConditionState.ACTIVE ? RED : PURPLE;
                conditionStatus[cond].setText(text);
                conditionStatus[cond].setForeground(color);
                conditionStatus[cond].setEnabled(true);
                conditionResetBtn[cond].setEnabled(true);
            }
            if (conditions != conditionMask) {
                updateConditionLayout(conditions);
            }
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(RED);
            periodTextField.setEnabled(false);
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
                conditionResetBtn[cond].setEnabled(false);
            }
        } 
    }

    private static final long serialVersionUID = 1L;
}
