package org.lsst.ccs.subsystem.ccob.thin;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeOutputLine;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearCapture;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MotorCommandListener;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.motorplatform.bus.SendAxisStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SendConfiguration;
import org.lsst.ccs.subsystem.motorplatform.bus.SendControllerStatus;
import org.lsst.ccs.subsystem.motorplatform.bus.SetupCapture;
import org.lsst.ccs.subsystem.motorplatform.bus.ShellCommandListener;
import org.lsst.ccs.subsystem.motorplatform.bus.StopAllMotion;

/**
 * Defines the names and other properties of the commands accepted by the subsystem. Delegates execution
 * of the commands to the component we're using to implement the {@code Controller} interface. That will
 * be either {@code FakeController} if the run mode is simulation, otherwise it will be {@code RealController}.
 * @author tether
 */
public class Commands extends Subsystem implements MotorCommandListener, ShellCommandListener {

    private static final Logger LOG = Logger.getLogger(Commands.class.getName());
    
    @LookupField(strategy=TREE)
    private volatile Controller controller;
    
    ////////// Lifecycle methods //////////

    public Commands() {
        super("commands", AgentInfo.AgentType.WORKER);
    }
    
    ////////// Commands specific to the thin beam CCOB //////////

    @Command(type=QUERY, level=NORMAL, description="(text) Returns the last reply from the TB server.")
    public String getLastReply() {return controller.getLastReply();}
    
    
    ////////// Implement MotorCommandListener and ShellCommandListener /////////

    @Command(level=NORMAL, description="(code) Relative motion of one axis.")
    @Override
    public void moveAxisRelative(final MoveAxisRelative cmd) {
        controller.moveAxisRelative(cmd);    
    }

    @Override
    @Command(level=NORMAL, description="(text) Relative motion: moveBy axisName positionChange speed")
    public void moveBy(final String axisName, final double positionChange, final double speed) {
        moveAxisRelative(
            new MoveAxisRelative(
                axisName,
                positionChange,
                Duration.ofMillis(Math.round(1000.0 * Math.abs(positionChange) / speed))
            )
        );
    }

    @Command(level=NORMAL, description="(code) Absolute motion of one axis.")
    @Override
    public void moveAxisAbsolute(final MoveAxisAbsolute cmd) {
        controller.moveAxisAbsolute(cmd);
    }

    @Command(level=NORMAL, description="(text) Absolute motion: moveTo axisName newPosition speed")
    @Override
    public void moveTo(final String axisName, final double newPosition, final double speed) {
        moveAxisAbsolute(new MoveAxisAbsolute(axisName, newPosition, speed));
    }

    @Command(type=QUERY, level=NORMAL,
        description="(code) Send ASAP on the status bus an AxisStatus message for the given axis.")
    @Override
    public void sendAxisStatus(final SendAxisStatus cmd) {
        controller.sendAxisStatus(cmd);
    }

    @Command(type=QUERY, level=NORMAL,
        description="(code) Send ASAP on the status bus a PlatformConfig message describing the hardware.")
    @Override
    public void sendConfiguration(final SendConfiguration cmd) {
        controller.sendConfiguration(cmd);
    }

    @Command(type=QUERY, level=NORMAL,
        description="(code) Send ASAP on the status bus a ControllerStatus message.")
    @Override
    public void sendControllerStatus(final SendControllerStatus cmd) {
        controller.sendControllerStatus(cmd);
    }

    @Command(level=NORMAL, description="(code) Stop any motion in progress.")
    @Override
    public void stopAllMotion(final StopAllMotion cmd) {
        controller.stopAllMotion(cmd);
    }

    @Command(level=NORMAL, description="(text) Stop any motion in progress. No arguments.")
    @Override
    public void stopAll() {
        stopAllMotion(new StopAllMotion());
    }

    @Command(type=QUERY, level=NORMAL,
        description="(text) Gets a text report on the status of a hardware controller.")
    @Override
    public String status(
        @Argument(description = "The controller name. One of XY, UB, P, H, K or SUMMARY. Case is ignored.")
        final String controllerName
    )
    {
        return controller.status(controllerName);
    }
    

    
    
    
    
    

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void changeAxisEnable(ChangeAxisEnable cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void changeOutputLine(ChangeOutputLine cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void clearAllFaults(ClearAllFaults cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void clearAxisFaults(ClearAxisFaults cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void clearCapture(ClearCapture cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void enableAllAxes(EnableAllAxes cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void disableAllAxes(DisableAllAxes cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void homeAxis(HomeAxis cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void setupCapture(SetupCapture cmd) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void clearFaults() {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void enable(String axisName) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void disable(String axisName) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    @Override
    @Command(level=NORMAL, autoAck=false, description="Not supported for thin-beam CCOB.")
    public void home(String axisName) {
        sendNack("Not supported for thin-beam CCOB.");
    }

    ////////// Photodiode commands //////////

    @Command(level = NORMAL, description="(text) Turns on the photodiode.")
    public void diodeOn() {
        controller.switchDiode("ON");
    }

    @Command(level = NORMAL, description="(text) Turns off the photodiode.")
    public void diodeOff() {
        controller.switchDiode("OFF");
    }
    
    @Command(level = NORMAL, description = "(text) Gets the current photodiode status.")
    public String diodeStatus() {
        return status("P");
    }
    
    ////////// Pico-ammeter commands //////////
    
    @Command(level = NORMAL, description="(text) Gets the current pico-ammeter status.")
    public String picoStatus() {return status("K");}
    
    @Command(level = NORMAL, description = "(text) Returns the pico-ammeter current in A.")
    public double picoReadCurrent() {
        return controller.picoReadCurrent();
    }
    
    @Command(level = NORMAL, description = "(text) Sets the pico-ammeter integration time.")
    public void picoSetTime(
        @Argument(description = "The integration time in ms.")
        final double millis
    )
    {
        controller.picoSetTime(millis);
    }
    
    @Command(level = NORMAL, description = "(text) Sets the expected current range for the pico-ammeter.")
    public void picoSetRange(
        @Argument(description = "The range index: -1 (auto-range) or 0 to 7 inclusive.")
        final int range
    )
    {
        controller.picoSetRange(range);
    }

    @Command(level = NORMAL, description = "(text) Gets the range setting in amps for the pico-ammeter.")
    public double picoGetRange() {
        return controller.picoGetRange();
    }
    
    ////////// Hyperchromator commands //////////

    @Command(level = NORMAL, description = "(text) Sets the light source wavelength.")
    public void hyperSetWavelength(
        @Argument(description = "Either a wavelength in nanometers or a color name like UVA or FIR."
            + " Case is ignored.")
        final String wavelength
    )
    {
        controller.hyperSetWavelength(wavelength);
    }

    @Command(level = NORMAL, description = "(text) Opens the high-speed shutter and leaves it open.")    
    public void hyperOpenFastShutter() {controller.hyperSwitchFastShutter("ON");}
    
    @Command(level = NORMAL, description = "(text) Closes the high-speed shutter and leaves it closed.")
    public void hyperCloseFastShutter() {controller.hyperSwitchFastShutter("OFF");}
    
    @Command(level = NORMAL, description = "(text) Opens the high-speed shutter, starts a timer then returns."
        + " The shutter will close when the timer expires but this command will not wait for that to happen.")
    public void hyperStartFastExposure(
        @Argument(description = "The number of milliseconds to keep the shutter open.")
        final int millis
    )
    {
        controller.hyperStartFastExposure(millis);
    }
    
    @Command(level = NORMAL, description = "(text) Opens the slow (main) shutter and leaves it open.")
    public void hyperOpenMainShutter() {controller.hyperSwitchMainShutter("ON");}
    
    @Command(level = NORMAL, description = "(text) Closes the slow (main) shutter and leaves it closed.")
    public void hyperCloseMainShutter() {controller.hyperSwitchMainShutter("OFF");}
    
    @Command(level = NORMAL, description = "(text) Turns on the light source and leaves it on.")
    public void hyperLightLamp() {controller.hyperSwitchLamp("ON");}
    
    @Command(level = NORMAL, description = "(text) Turns off the light source and leaves it off.")
    public void hyperKillLamp() {controller.hyperSwitchLamp("OFF");}
    
    @Command(level = NORMAL, description = "(text) Removes the filter and moves to the zero-order position"
        + " of one of the gratings.")
    public void hyperRemoveFilter(
        @Argument(description = "The name of the grating, A or B. case is ignored.")
        final String gratingName)
    {
        controller.hyperZOP(gratingName);
    }
    
    @Command(level = NORMAL, description = "(text) Returns a status report for the hyperchromator.")
    public String hyperStatus() {return status("H");}
    
    ////////// Combined hyperchromator and pico-ammeter commands //////////
    
    @Command(level = NORMAL, description = "(text) Starts the pico-ammeter reading after opening the high-speed"
        + " shutter. Returns the reading in amps. Preferred over readThenIlluminate().")
    public double illuminateThenRead(
        @Argument(description = "The pico-ammeter integration time, and the high-speed shutter time to be open,"
            + " in ms.")
        final int millis
    )
    {
        return controller.illuminateThenRead(millis);
    }
    
    @Command(level = NORMAL, description = "(text) Starts the pico-ammeter reading before opening"
        + " the high-speed shutter. The pico-ammeter"
            + " range and integration time must have been previously set. "
        + "Auto-ranging not allowed. Returns the reading in amps. May have sync problems.")
    public double readThenIlluminate(
        @Argument(description = "The time to keep the high-speed shutter open, in ms.")
        final int millis
    )
    {
        return controller.readThenIlluminate(millis);
    }
    
    @Command(level = NORMAL, description = "(text) Gets the diode repsonse attenuation for the current"
        + " wavelength setting of the hyperchromator (200-11nm only).")
    public double getDiodeAttenuation() {return controller.getDiodeAttenuation();}
    
    ////////// Targeting commands //////////
    
    @Command(level = NORMAL, description = "(text) Displays information about the current target.")
    public String getTarget() {return controller.getTarget();}
    
    @Command(level = NORMAL, description = "(text) Sets the target to the given sensor plane XY coordinates.")
    public void setTargetTo(
        @Argument(description = "The desired sensor plane X coordinate, in mm.")
        final double x,
        @Argument(description = "The desired sensor plane Y coordinate, in mm.")
        final double y
    )
    {
        controller.setTargetTo(x, y);
    }
    
    @Command(level = NORMAL, description = "(text) Sets the target to the current sensor plane coordinates.")
    public void setTargetHere() {controller.setTargetHere();}
    
    @Command(level = NORMAL, description = "(text) Move in XY to aim at current target after a change in UB.")
    public void aimAgainXY() {controller.aimAgainXY();}
    
    @Command(level = NORMAL, description = "(text) Move in UB to aim at current target after a change in XY.")
    public void aimAgainUB() {controller.aimAgainUB();}
    
    
}
