package org.lsst.ccs.subsystem.ccob.thin;

import org.lsst.ccs.subsystem.ccob.thin.buses.TBServerException;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy;

/**
 * Defines the alerts raised by the subsystem.
 * @author tether
 */
public class Alerts {
    
    private Alerts() {}
    
    /**
     * We raise this alert when the TB_Server replies to a command with an error
     * or warning message. The cause text we use will be the reply from TB_Server. Since each reply
     * might be different, we'll use the {@code RaiseAlertStrategy ALWAYS}.
     */
    public static Alert SERVER_ERROR =
        new Alert("SERVER_ERROR", "TB_Server replied with an error or a warning.");
    
    /**
     * We raise this alert when the subsystem tried and failed to send a command to TB_Server.
     * The cause text we use will be the summary message from the {@code DriverException}.
     * Since comm problems may persist for more than a few seconds and since we poll for
     * status every one or two seconds, we'll use the {@code RaiseAlertStrategy ON_SEVERITY_CHANGE}.
     */
    public static Alert COMM_ERROR =
        new Alert("COMM_ERROR", "There is a problem communicating with TB_Server.");
    
    /**
     * We raise this alert when none of the other alerts are applicable. The cause text we use
     * will be the summary message from the exception. Since this kind of problem will almost certainly
     * not go away without code fixes, we'll use the {@code RaiseAlertStrategy ON_SEVERITY_CHANGE}.
     * 
     */
    public static Alert INTERNAL_ERROR =
        new Alert("INTERNAL_ERROR", "An exception was raised indicating a problem with the subsystem.");
    
    /**
     * Registers all the alerts defined in this class. For all of them the clear-alert handler
     * is set to {@code AlertService.ALWAYS}.
     * @param al an instance of the alert service.
     */
    public static void registerAllAlerts(final AlertService al) {
        al.registerAlert(SERVER_ERROR, AlertService.ALWAYS);
        al.registerAlert(COMM_ERROR, AlertService.ALWAYS);
        al.registerAlert(INTERNAL_ERROR, AlertService.ALWAYS);
    } 

    /**
     * Raises a server alert.
     * @param al an instance of the alert service.
     * @param exc the exception containing the server's error message.
     * @see SERVER_ERROR
     */
    public static void raiseServerAlert(final AlertService al, final TBServerException exc) {
        al.raiseAlert(SERVER_ERROR, AlertState.ALARM, exc.getMessage(), RaiseAlertStrategy.ALWAYS);
    }
    
    /**
     * Raises a comm alert.
     * @param al an instance of the alert service.
     * @param exc the exception raised by the comm driver.
     * @see COMM_ERROR
     */
    public static void raiseCommAlert(final AlertService al, final DriverException exc) {
        final Throwable cause = exc.getCause() != null ? exc.getCause() : exc;
        al.raiseAlert(
            COMM_ERROR, AlertState.WARNING, exc.getMessage(), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
    }
    
    /**
     * Raises an internal error alert.
     * @param al an instance of the alert service.
     * @param exc the exception raised by the subsystem code or the Java runtime.
     * @see INTERNAL_ERROR
     */
    public static void raiseInternalAlert(final AlertService al, final Throwable exc) {
        al.raiseAlert(
            INTERNAL_ERROR, AlertState.ALARM, exc.getMessage(), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
    }
}
