package org.lsst.ccs.subsystem.cablerotator;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A quick and dirty simulation of an simple Aerotech controller with one axis named THETA.
 * @author tether
 */
public class MockAscii extends Ascii {

    private int planeStatus;
    private int xStatus, xFault;
    private double xPos;
    private int taskStatus;
    private Instant stopTime;
    private final Map<Integer, Double> dglobal;
    private final Map<Integer, Integer> iglobal;

    public MockAscii() {
        this.planeStatus = 0;
        this.xStatus = 3; // Enabled, homed.
        this.xFault = 0;
        this.xPos = 0.0;
        this.taskStatus = 0;
        this.stopTime = null;
        this.dglobal = new HashMap<>();
        this.iglobal = new HashMap<>();
    }

    private static final Logger LOG = Logger.getLogger(MockAscii.class.getName());

    @Override
    public synchronized void setTimeout(int time) throws DriverException {
    }

    @Override
    public void setTimeout(double time) throws DriverException {
    }

    @Override
    public void setTerminator(Terminator term) {
    }

    @Override
    public void setCommandTerm(Terminator term) {
    }

    @Override
    public void setResponseTerm(Terminator term) {
    }

    @Override
    public synchronized void flush() throws DriverException {
    }

    @Override
    public int readBytes(byte[] buff, int offset, int leng) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public int readBytes(byte[] buff, int offset) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void writeBytes(byte[] command, int offset, int leng) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void writeBytes(byte[] command) throws DriverException {
        throw new UnsupportedOperationException("Not implemented");
    }


    @Override
    public synchronized String read(String command) throws DriverException {
        updateStatus(); // Status updates occur with each command. Assumes that at least a status query happens regularly.
        if (command.startsWith("DGLOBAL")) {
            return doDglobal(command);
        }
        else if (command.startsWith("IGLOBAL")) {
            return doIglobal(command);
        }
        else if (command.startsWith("AXISSTATUS")) {
            return doAxisStatus(command);
        }
        else if (command.startsWith("AXISFAULT")) {
            return doAxisFault(command);
        }
        else if (command.startsWith("TASKSTATE")) {
            return doTaskStatus(command);
        }
        else if (command.contains("Move.bcx")) {
            return doMove(command);
        }
        else if (command.contains("HomeAsync.bcx")) {
            return doHome(command);
        }
        else if (command.startsWith("PROGRAM STOP")) {
            return doStop(command);
        }
        else if (command.startsWith("ABORT")) {
            return doAbort(command);
        }
        else if (command.startsWith("PLANESTATUS")) {
            return doPlaneStatus(command);
        }
        else if (command.startsWith("PFBK")) {
            return doPositionQuery(command);
        }
        else {
            // Unknown command. Return "%" for success followed by a zero in case a value is expected.
            LOG.info(command);
            return "%0";
        }
    }

    @Override
    public synchronized String read() throws DriverException {
        LOG.info("Read()");
        return "%0";
    }

    @Override
    public synchronized void write(String command) throws DriverException {
        LOG.info("Write(): " + command);
    }

    @Override
    public boolean closeSilent() {
        return true;
    }

    @Override
    public synchronized void close() throws DriverException {
    }

    @Override
    public void openFtdi(String serialNo, int baudRate) throws DriverException {
    }

    @Override
    public void openFtdi(String serialNo, int baudRate, int dataChar) throws DriverException {
    }

    @Override
    public void openSerial(String devcName, int baudRate) throws DriverException {
    }

    @Override
    public void openSerial(String devcName, int baudRate, int dataChar) throws DriverException {
    }

    @Override
    public void openNet(String host, int port) throws DriverException {
    }

    @Override
    public void open(ConnType type, String ident, int parm) throws DriverException {
    }

    @Override
    public synchronized void open(ConnType type, String ident, int parm1, int parm2) throws DriverException {
    }

    private static class ParsedGlobal {
        public final int index;
        public final String value;
        ParsedGlobal(final String command) {
            LOG.info(command);
            final int eqpos = command.indexOf("=");
            final int lpos = command.indexOf("(");
            final int rpos = command.indexOf(")");
            index = Integer.valueOf(command.substring(lpos+1, rpos));
            if (eqpos > 0) {
                value = command.substring(eqpos+1);
            }
            else {
                value = null;
            }
        }
    }

    private void updateStatus() {
        if (stopTime != null && Instant.now().isAfter(stopTime)) {
            // We are no longer moving.
            stopTime = null;
            taskStatus = 0;
            planeStatus = 0;
            xStatus = 3;
            dglobal.put(0, 0.0); // Completion code = OK.
        }
    }

    private String doDglobal(final String command) {
        final ParsedGlobal pg = new ParsedGlobal(command);
        if (pg.value != null) {
            dglobal.put(pg.index, Double.valueOf(pg.value));
            return "%";
        }
        else {
            return "%" + getDglobal(pg.index);
        }
    }

    private Double getDglobal(final int index) {
        Double result = dglobal.get(index);
        if (result == null) result = 0.0;
        return result;
    }

    private String doIglobal(final String command) {
        final ParsedGlobal pg = new ParsedGlobal(command);
        if (pg.value != null) {
            iglobal.put(pg.index, Integer.valueOf(pg.value));
            return "%";
        }
        else {
            return "%" + getIglobal(pg.index);
        }
    }

    private Integer getIglobal(final int index) {
        Integer result = iglobal.get(index);
        if (result == null) result = 0;
        return result;
    }

    private String doAxisStatus(final String command) {
        int stat;
        LOG.info(command);
        if (command.contains("(THETA)")) {
            return "%" + xStatus;
        }
        else {
            return "!"; // Bad axis name.
        }
    }

    private String doAxisFault(final String command) {
        LOG.info(command);
        if (command.contains("(THETA)")) {
            return "%" + xFault;
        }
        else {
            return "!";
        }
    }

    private String doPlaneStatus(final String command) {
        LOG.info(command);
        return "%" + planeStatus;
    }

    private String doTaskStatus(final String command) {
        LOG.info(command);
        return "%" + taskStatus;
    }

    private String doStop(final String command) {
        LOG.info(command);
        stopTime = null;
        xStatus = 3;
        planeStatus = 0;
        taskStatus = 0;
        dglobal.put(0, 0.0);
        return "%";
    }

    private String doAbort(final String command) {
        LOG.info(command);
        stopTime = null;
        xStatus = 3;
        planeStatus = 0;
        taskStatus = 0;
        dglobal.put(0, 0.0);
        return "%";
    }

    private String doMove(final String command) {
        LOG.info(command);
        final int axisNum = getDglobal(0).intValue();
        final long moveMillis = (long)Math.min(10000.0, 800.0 * getDglobal(4));
        final double newPos = getDglobal(1).doubleValue();
        if (axisNum == 0) {
            stopTime = Instant.now().plusMillis(moveMillis);
            taskStatus = 3; // Running.
            xStatus |= (1 << 3); // Moving.
            xPos = newPos;
        }
        return "%";
    }

    private String doHome(final String command) {
        LOG.info(command);
        final int axisNum = getIglobal(32);
        taskStatus = 3;
        if (axisNum == 0) {
            xStatus |= (1 << 3);
            stopTime = Instant.now().plusSeconds(10);
        }
        return "%";
    }

    private String doPositionQuery(final String command) {
        // We don't distinguish between raw, program and calibrated positions.
        LOG.info(command);
        if (command.contains("(THETA)")) {
            return "%" + xPos;
        }
        else {
            return "!";
        }
    }
}
