package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A trendable record for Lighthouse OPC instruments. Can hold multiple items but in
 * practice we use just one.
 * @author tether
 */
public class TrendableOPCRecord  implements TrendableRecord {

    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Serializable> items;

    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param items A {@code Map} from value key to serializable value.
     */
    public TrendableOPCRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = java.util.Collections.unmodifiableMap(items);
    }

    /** {@inheritDoc } */
    @Override
    public String getMasterKey() {return masterKey;}

    /** {@inheritDoc } */
    @Override
    public Instant getMasterTimestamp() {return masterTimestamp;}

    /** {@inheritDoc } Alerts are manufactured from limit violation and malfunction flags. */
    @Override
    public List<Alert> getAlerts() {
        final List<Alert> alerts = new ArrayList<>(30);
        items.entrySet().stream()
            .forEach(x -> {
                if (x.getValue() instanceof CounterPoint) {
                    final CounterPoint cp = (CounterPoint) x.getValue();
                    if (cp.limitViolation != 0) {
                        alerts.add(Alerts.limitViolationAlert(masterKey, x.getKey()));
                    }
                    if (cp.malfunction != 0) {
                        alerts.add(Alerts.instrumentMalfunctionAlert(masterKey, x.getKey()));
                    }
                }
                if (x.getValue() instanceof AnalogPoint) {
                    final AnalogPoint ap = (AnalogPoint) x.getValue();
                    if (ap.limitViolation != 0) {
                        alerts.add(Alerts.limitViolationAlert(masterKey, x.getKey()));
                    }
                    if (ap.malfunction != 0) {
                        alerts.add(Alerts.instrumentMalfunctionAlert(masterKey, x.getKey()));
                    }
                }
            });
        return alerts;
    }

    /** {@inheritDoc } */
    @Override
    public Map<String, Serializable> getItems() {return items;}

    /** {@inheritDoc } */
    @Override
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey,
                CCSTimeStamp.currentTimeFromMillis(masterTimestamp.toEpochMilli()));
        items.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }

    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }

}
