package org.lsst.ccs.subsystem.airwatch.main;


import java.io.Serializable;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;

/**
 * The trendable record for dummy instruments. Doesn't actually post to the status bus, just writes to the log.
 * @author tether
 */
public class TrendableDummyRecord implements TrendableRecord {

    private static final Logger LOG = Logger.getLogger(TrendableDummyRecord.class.getName());
    
    private final String masterKey;

    private final Instant masterTimestamp;
        
    private final Map<String, Serializable> items;
    
    public TrendableDummyRecord(
        String masterKey, Instant masterTimestamp, Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = new TreeMap<>(items);
    }
    
    
    
    @Override
    public String getMasterKey() {
        return masterKey;
    }

    @Override
    public Instant getMasterTimestamp() {
        return masterTimestamp;
    }

    @Override
    public List<Alert> getAlerts() {
        return Collections.emptyList();
    }

    @Override
    public Map<String, Serializable> getItems() {
        return Collections.unmodifiableMap(items);
    }

    @Override
    public void post(Subsystem subsys) {
        LOG.log(Level.INFO, "DUMMY {0} at {1}", new Object[]{masterKey, masterTimestamp});
        for (final String key: items.keySet()) {
            LOG.log(Level.INFO, "DUMMY {0} = {1}", new Object[]{key, items.get(key).toString()});
        }
    }

    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }

}
