package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.opc.OPCClient;
import org.lsst.ccs.drivers.opc.OPCItem;
import org.lsst.ccs.drivers.opc.Quality;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.COUNTER_0_3;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.COUNTER_3_0;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.HUMIDITY;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.TEMPERATURE;
import org.lsst.ccs.utilities.exc.BundledException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Represents an instrument connected to PC running Lighthouse's LMS Express RT program. That program
 * implements an OPC DA server holding only the latest readings from each instrument connected
 * to LMS Express RT.
 * <p>
 * Each instrument is a cluster of several sensors: two particle counters (for different-sized particles),
 * an ambient relative humidity sensor and an ambient temperature sensor. Each sensor is represented
 * in the instrument by its own "channel". The PC periodically directs each instrument to take
 * sensor readings. Each channel produces several values which are then published by the PC using the 
 * OPC DA protocol. Due to the way the protocol works each value is published with its own
 * timestamp even though all the values for a given channel are produced at the same time.
 * <p>
 * Each counter channel produces an absolute particle count for the air sample, a particle density,
 * a flag indicating a limit violation (out of bounds value) and a sensor malfunction flag. Each
 * analog channel produces a single value together with limit violation and malfunction flags.
 * <p>
 * Reading the latest OPC DA values using this class produces series of "trendable records".
 * One trendable record is produced per channel. Each has a master key which is the short name for
 * the instrument's location and a master timestamp which is the OPC DA timestamp of one of the
 * channel's values. The trendable record carries the sensor values as a single "item" with
 * a key which is the channel name and a serializable object which is an instance of either {@code AnalogPoint}
 * or {@code CounterPoint} depending on the type of channel. For counter channels the name
 * is the particle size in microns, e.g., "3.0", "0.3". Temperature channels are named "temp"
 * and humidity channels "humid".
 * <p>
 * A trendable record record is published on the CCS data bus as a KeyValueDataList (KVD list) whose
 * master timestamp and key are those of the record. Each KVD list has a single member derived from
 * the key and value of the single item in each trendable record. Therefore the trending path
 * to each value is composed of the agent name, the short sensor name, the channel name and
 * the name of the field from the {@code AnalogPoint} or {@code CounterPoint} object. For example
 * ir2-airwatch/MAIN_NW/3.0/density. Each published KVDL list looks like this:
 * <ul>
 *     <li>Master key = short location name</li>
 *     <li>Master timestamp</li>
 *     <li>Item list
 *         <ul>
 *             <li>Item key = channel name, value = instance of AnalogPoint or CounterPoint</li>
 *         </ul>
 *     </li>
 * </ul>
 * @see AnalogPoint
 * @see CounterPoint
 * @see TrendableOPCRecord
 * @author tether
 */
public class LighthouseOPC implements Instrument {
    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");

    // Timeout for I/O operations.
    private static final Duration DEVICE_TIMEOUT = Duration.ofSeconds(10);

    // Let the driver log debug messages?
    private static final boolean DEVICE_DEBUG = false;

    private final int index;
    private final boolean enabled;
    private final String location;
    private final String shortName;
    private final Instant lastDataTime;
    private final String connectionInfo;
    private final Optional<HardwareException> lastException;
    private final List<TrendableRecord> trendables;
    private final LocalConfigService config;
    private final AlertService alertService;

    /**
     * Constructs a new instance from configuration data. The new instance will
     * have no stored data and have no stored exception.
     * @param iconf The subsystem's configuration information for the instrument.
     */
    public LighthouseOPC(InstrumentConfig iconf) {
        this.index = iconf.index;
        this.enabled = iconf.working;
        this.location = iconf.loc;
        this.shortName = iconf.shortName;
        this.lastDataTime = Instant.MIN;
        this.connectionInfo = iconf.conn;
        this.lastException = Optional.empty();
        this.trendables = Collections.emptyList();
        this.config = iconf.localConfig;
        this.alertService = iconf.alertService;
    }

    /**
     * Constructs a new instance using a value specified for each field.
     * @param index The instrument index (place in the list of instruments).
     * @param enabled The enabled-for-readout flag.
     * @param location The name of the location being monitored.
     * @param shortName The short name for the instrument.
     * @param lastDataTime The most recent timestamp seen.
     * @param connectionInfo The string of connection info from the configuration.
     * @param lastException The last exception thrown by an operation.
     * @param trendables The trendable items resulting from the last readout.
     * @param config The configuration localConfig ref.
     * @param alertService A ref to the alert service.
     */
    private LighthouseOPC(
        int index,
        boolean enabled,
        String location,
        String shortName,
        Instant lastDataTime,
        String connectionInfo,
        Optional<HardwareException> lastException,
        List<TrendableRecord> trendables,
        LocalConfigService config,
        AlertService alertService)

    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.shortName = shortName;
        this.lastDataTime = lastDataTime;
        this.connectionInfo = connectionInfo;
        this.lastException = lastException;
        this.trendables = trendables;
        this.config = config;
        this.alertService = alertService;
    }

    /** {@inheritDoc } */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastException;
    }

    /** {@inheritDoc } */
    @Override
    public int getIndex() {
        return index;
    }

    /** {@inheritDoc } */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, shortName, lastDataTime);
    }

    /** {@inheritDoc } */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return trendables.stream();
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        return new LighthouseOPC(
            index,
            false,
            location,
            shortName,
            lastDataTime,
            connectionInfo,
            lastException,
            trendables,
            config,
            alertService);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument enable() {
        final boolean newEnabled = true;
        return new LighthouseOPC(
            index,
            newEnabled,
            location,
            shortName,
            lastDataTime,
            connectionInfo,
            Optional.empty(),
            Collections.emptyList(),
            config,
            alertService);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument read() {
        HardwareException hwexc = null;
        List<TrendableRecord> newData = Collections.emptyList();
        Instant newLast = lastDataTime;
        Instrument ins;
        OPCClient client = null;
        try {
            client = parseConnectionInfo();
            final ReadResults res = getLatestReadings(client);
            client = res.client;
            newData = res.data;
            if (client.getException().isPresent()) {
               throw client.getException().get();
            }
        }
        catch (DriverException exc) {
            hwexc = new HardwareException(true, exc);
        }
        catch (Throwable exc) {
            throw new RuntimeException(exc);
        }
        finally {
            if (client != null) {client.close();}
        }

        // If the read was successful, update read time.
        if (newData.size() > 0) {
            newLast = Instant.now();
        }
        ins = new LighthouseOPC(
            index,
            enabled,
            location,
            shortName,
            newLast,
            connectionInfo,
            Optional.ofNullable(hwexc),
            Collections.unmodifiableList(newData),
            config,
            alertService);
        return ins;
    }

    // Interpret the connection info string to yield an OPCClient object.
    private OPCClient parseConnectionInfo() throws DriverException {
        // The connection info is the name of a CCS configuration item. That item
        // is an array of strings containing this information:
        // (0) The host name or IPv4 address for the server.
        // (1) The Windows domain of the local account used to log in.
        // (2) The class ID (CLSID) of the OPC server.
        // (3) The username of the Windows local account used to log in.
        // (4) The password of the Windows local account used to log in.
        // In the usual case all the "instruments" will actually be on the same server.
        final List<String> srvInf = config.getOPCServerInfo(connectionInfo);
        if (srvInf.size() < 5) {
            throw new DriverException("Login information for "
                                      + connectionInfo + " is missing or incomplete.");
        }
        return new OPCClient(srvInf.get(0), srvInf.get(1), srvInf.get(2),
            srvInf.get(3), srvInf.get(4));
    }

    private ReadResults getLatestReadings(final OPCClient oldClient)
    throws Throwable
    {
        OPCClient client = oldClient;
        // Construct the OPC tag prefix for this location.
        final String prefix = "LMS/Inputs/" + location + "/";

        // Construct all the tags we want to read..

        // There is a separate item produced for each tag in the group. Note that the
        // short location name has its own item, as do the value and the flags for each
        // channel.

        // Although tags for particle count alarm limits are also defined, they read as zero.
        // This may be a general defect or a result of using the "3 bad of last 5" trigger mode
        // for the alarm.

        final String loc = prefix + "Short Location Name";

        final String abs30   = prefix + "3.0/Aggregates/Cumulative Counts/Data";
        final String norm30  = prefix + "3.0/Aggregates/Cumulative Normalized Counts/Data";
        final String fault30 = prefix + "3.0/Aggregates/Cumulative Counts/Status/Fault/Triggered";
        final String alarm30 = prefix + "3.0/Aggregates/Cumulative Counts/Status/Alarm/Triggered";

        final String abs03   = prefix + "0.3/Aggregates/Cumulative Counts/Data";
        final String norm03  = prefix + "0.3/Aggregates/Cumulative Normalized Counts/Data";
        final String fault03 = prefix + "0.3/Aggregates/Cumulative Counts/Status/Fault/Triggered";
        final String alarm03 = prefix + "0.3/Aggregates/Cumulative Counts/Status/Alarm/Triggered";

        final String humidity       = prefix + "Relative Humidity/Data";
        final String faultHumid     = prefix + "Relative Humidity/Status/Fault/Triggered";
        final String alarmHumid     = prefix + "Relative Humidity/Status/Alarm/Triggered";
        final String alarmHumidLow  = prefix + "Relative Humidity/Status/Alarm/Low Limit Value";
        final String alarmHumidHigh = prefix + "Relative Humidity/Status/Alarm/High Limit Value";

        final String temperature    = prefix + "Temperature/Data";
        final String faultTemp      = prefix + "Temperature/Status/Fault/Triggered";
        final String alarmTemp      = prefix + "Temperature/Status/Alarm/Triggered";
        final String alarmTempLow   = prefix + "Temperature/Status/Alarm/Low Limit Value";
        final String alarmTempHigh  = prefix + "Temperature/Status/Alarm/High Limit Value";

        final List<String> tags =
            Arrays.asList(new String[]{loc,
                                       abs30, norm30, fault30, alarm30,
                                       abs03, norm03, fault03, alarm03,
                                       humidity, faultHumid, alarmHumid,
                                       alarmHumidLow, alarmHumidHigh,
                                       temperature, faultTemp, alarmTemp,
                                       alarmTempLow, alarmTempHigh});
        // Make a group from the tags and read the group. It's unfortunate but
        // each addition to the group sends a validity-check request to
        // the server. The same behavior prevents us from making a group
        // in advance without a server connection. For only a few channels
        // read out at long intervals the overhead is tolerable.
        client =
            client
                .setGroup(tags)
                .readGroup(true);
        
        // Map the OPC items by tag name.
        final Map<String, OPCItem> serverItems =
            client.getData()
            .stream()
            .collect(Collectors.toMap(x -> x.getTag(), x -> x));

        // Check whether we got the right number of items. Make a discrepancy look
        // like a device error.
        if (serverItems.size() != tags.size()) {
            throw new DriverException(
                String.format("Wrong no. of items read; %d instead of %d",
                    serverItems.size(),
                    tags.size()));
        }
        // All items must be of GOOD quality.
        BundledException exc = null;
        for (final String tag: serverItems.keySet()) {
            if (serverItems.get(tag).getQuality() != Quality.GOOD) {
                final String msg = String.format("Marked by server as not good: %s", tag);
                exc = new BundledException(msg, exc);
            }
        }
        if (exc != null) {
            throw new DriverException(exc);
        }

        // Get the short location name that was read.
        final String shortLocName = (String)serverItems.get(loc).getValue();

        // Generate alerts for limit violations and malfunctions in analog sensors.
        checkAnalogFlags(shortLocName, "Temperature", "C",
             serverItems, temperature, alarmTemp, alarmTempLow, alarmTempHigh, faultTemp);
        checkAnalogFlags(shortLocName, "Humidity", "%",
            serverItems, humidity, alarmHumid, alarmHumidLow, alarmHumidHigh, faultHumid);

        // Generate alerts for high counts or malfunctions in particle counters.
        checkCounterFlags("0.3", shortLocName, serverItems, abs03, alarm03, fault03);
        checkCounterFlags("3.0", shortLocName, serverItems, abs30, alarm30, fault30);

        // Convert the remaining OPCItems into TrendableRecords. For this we'll need to replace
        // the server path name of each data item with the short name used to
        // make the trending tag.

        // For each channel assemble one TrendableRecord from the short location name
        // and and the items read for the channel.
        final List<TrendableRecord> result = new ArrayList<>();
        result.add(makeCounterRecord(serverItems, abs30, norm30, fault30, alarm30,
                                     COUNTER_3_0, shortLocName));
        result.add(makeCounterRecord(serverItems, abs03, norm03, fault03, alarm03,
                                     COUNTER_0_3, shortLocName));
        result.add(makeAnalogRecord(serverItems, humidity, faultHumid, alarmHumid,
                                     HUMIDITY, shortLocName));
        result.add(makeAnalogRecord(serverItems, temperature, faultTemp, alarmTemp,
                                     TEMPERATURE, shortLocName));
        return new ReadResults(client, result);
    }

    private void checkAnalogFlags(
        final String shortLocName,
        final String quantity,
        final String units,
        final Map<String, OPCItem> serverItems,
        final String valueTag,
        final String alarmTag,
        final String alarmTagLow,
        final String alarmTagHigh,
        final String faultTag)
    {
        final boolean hasAlarm  = (Boolean)serverItems.get(alarmTag).getValue();
        if (hasAlarm) {
            final double value     = (Double)serverItems.get(valueTag).getValue();
            final double alarmLow  = (Double)serverItems.get(alarmTagLow).getValue();
            final double alarmHigh = (Double)serverItems.get(alarmTagHigh).getValue();
            final Alert alert = Alerts.limitViolationAlert(shortLocName, quantity);
            final String causeFormat = "%s at %s of %.1f%s is outside of [%.1f%s, %.1f%s].";
            final String cause = String.format(causeFormat,
                quantity, shortLocName, value, units, alarmLow, units, alarmHigh, units);
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
        }
        final boolean hasFault = (Boolean)serverItems.get(faultTag).getValue();
        if (hasFault) {
            final Alert alert = Alerts.instrumentMalfunctionAlert(shortLocName, quantity);
            final String causeFormat = "%s sensor at %s reports a malfunction.";
            final String cause = String.format(causeFormat, quantity, shortLocName);
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
        }
    }

    private void checkCounterFlags(
        final String particleSize,
        final String shortLocName,
        final Map<String, OPCItem> serverItems,
        final String valueTag,
        final String alarmTag,
        final String faultTag)
    {
        final boolean isAlarm  = (Boolean)serverItems.get(alarmTag).getValue();
        if (isAlarm) {
            final double value     = (Double)serverItems.get(valueTag).getValue();
            final Alert alert = Alerts.limitViolationAlert(shortLocName, particleSize);
            final String causeFormat = "%s-micron particle count at %s of %g is too high.";
            final String cause = String.format(causeFormat,
                particleSize, shortLocName, value);
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
        }
        final boolean hasFault = (Boolean)serverItems.get(faultTag).getValue();
        if (hasFault) {
            final Alert alert = Alerts.instrumentMalfunctionAlert(shortLocName, "Counter");
            final String causeFormat = "Particle counter at %s reports a malfunction.";
            final String cause = String.format(causeFormat, shortLocName);
            alertService.raiseAlert(alert, AlertState.ALARM, cause);
        }
    }

    private static class ReadResults {
        public final OPCClient client;
        public final List<TrendableRecord> data;
        ReadResults(final OPCClient client, final List<TrendableRecord> data) {
            this.client = client;
            this.data = data;
        }
    }

    private TrendableRecord makeCounterRecord(
        final Map<String, OPCItem> serverItems,
        final String absoluteKey,
        final String densityKey,
        final String faultKey,
        final String alarmKey,
        final InstrumentChannel channel,
        final String shortLocName)
    {
        final Map<String, Serializable> trendingItems = new TreeMap<>();
        // Although the count, alarm flag and malfunction flag each have their
        // own timestamp, they should all be about the same so we'll use the
        // timestamp of the count for all of them. In the trending display
        // of the CCS graphical console we want each particle size to show
        // as a subtree item with count and flags as leaf items underneath.
        // To do that we associate the channel-name tag with an object
        // containing the items s read for that channel.
        trendingItems.put(channel.getKey(), // E.g., "0.3", "3.0"
            new CounterPoint(
                (Double)serverItems.get(absoluteKey).getValue(), // particles in sample
                (Double)serverItems.get(densityKey).getValue(),  // particles per unit volume
                (Boolean)serverItems.get(alarmKey).getValue(),    // limitViolation flag
                (Boolean)serverItems.get(faultKey).getValue()     // malfunction flag
            )
        );
        return new TrendableOPCRecord(
            shortLocName,
            serverItems.get(absoluteKey).getTimestamp(), // time of channel count.
            trendingItems
        );
    }

    // See the similar code for counter channels above.
    private TrendableRecord makeAnalogRecord(
        final Map<String, OPCItem> serverItems,
        final String valueKey,
        final String faultKey,
        final String alarmKey,
        final InstrumentChannel channel,
        final String shortLocName)
    {
        final Map<String, Serializable> trendingItems = new TreeMap<>();
        trendingItems.put(channel.getKey(),
            new AnalogPoint(
                (Double)serverItems.get(valueKey).getValue(),
                (Boolean)serverItems.get(alarmKey).getValue(),
                (Boolean)serverItems.get(faultKey).getValue()
            ));
        return new TrendableOPCRecord(
            shortLocName,
            serverItems.get(valueKey).getTimestamp(),
            trendingItems
        );
    }
}
