package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import java.util.stream.Stream;
import org.lsst.ccs.HardwareException;

/**
 * A dummy instrument used to test the subsystem. Produces a trendable cosine wave of unit amplitude on the
 * 1.0-micron particle channel. Every fifth simulated operation fails. Read operations take ten
 * seconds unless interrupted or unless they experience a simulated exception.
 *
 * @author tether
 */
public class DummyInstrument implements Instrument {

    private final int index;
    private final boolean enabled;
    private final String location;
    private final String shortName;
    private final Instant lastDataTime;
    private final long nops;
    private final Optional<HardwareException> lastExc;
    private final List<TrendableRecord> lastData;

    public DummyInstrument(InstrumentConfig config) {
        this.index = config.index;
        this.enabled = config.working;
        this.location = config.loc;
        this.shortName = config.shortName;
        this.lastDataTime = Instant.MIN;
        this.nops = 0;
        this.lastExc = Optional.empty();
        this.lastData = Collections.emptyList();
    }

    private DummyInstrument(
            int index,
            boolean enabled,
            String location,
            String shortName,
            Instant lastDataTime,
            long nops,
            Optional<HardwareException> exc,
            List<TrendableRecord> data)
    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.shortName = shortName;
        this.lastDataTime = lastDataTime;
        this.nops = nops;
        this.lastExc = exc;
        this.lastData = data;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastExc;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public int getIndex() {
        return index;
    }

    /**
     * {@inheritDoc }
     */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, shortName, lastDataTime);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return lastData.stream();
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        final long n = nops + 1;
        final Optional<HardwareException> exc =
                (n % 5 == 0)
                ? Optional.of(new HardwareException("Simulated exception during disable.", null))
                : Optional.empty();
        return new DummyInstrument(index, false, location, shortName, lastDataTime, n, exc, lastData);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument enable() {
        final long n = nops + 1;
        final Optional<HardwareException> exc =
                (n % 5 == 0)
                ? Optional.of(new HardwareException("Simulated exception during enable.", null))
                : Optional.empty();
        return new DummyInstrument(index, true, location, shortName, lastDataTime, n, exc, lastData);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument read() {
        // Simulate a hardware exception if it's time for one.
        final long n = nops + 1;
        if (n % 5 == 0) {
            return new DummyInstrument(index, enabled, location, shortName, lastDataTime, n,
            Optional.of(new HardwareException(false, "Simulated exception during read.")),
            Collections.emptyList());
        }

        // Simulate a time-consuming operation, which if interrupted becomes
        // essentially a no-op.
        try {
            MILLISECONDS.sleep(100);
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt(); // Caller must know about interruptions.
            return new DummyInstrument(index, enabled, location, shortName, lastDataTime, n,
            Optional.empty(), Collections.emptyList());
        }

        // Generate data. We give each sensor its own trendable record with a single item, just as is
        // done for the OPC instruments.
        final List<TrendableRecord> tr = new ArrayList<>();
        final Instant now = Instant.now();
        final double density = Math.cos(now.getEpochSecond());
        Map<String, Serializable> items = new TreeMap<>();
        items.put(InstrumentChannel.COUNTER_1_0.getKey(), new CounterPoint(density * 10.0, density, (density > 0.8), false));
        tr.add(new TrendableDummyRecord(location, now, items));
        items.clear();
        items.put(InstrumentChannel.TEMPERATURE.getKey(), new AnalogPoint(30.0 /* deg C */, false, false));
        tr.add(new TrendableDummyRecord(location, now, items));
        items.clear();
        items.put(InstrumentChannel.HUMIDITY.getKey(), new AnalogPoint(60.0 /* % */, false, false));
        tr.add(new TrendableDummyRecord(location, now, items));
        return new DummyInstrument(index, enabled, location, shortName, now, n,
            Optional.empty(), tr);
    }
}
