package org.lsst.ccs.subsystem.airwatch.main;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Implements a local configuration service based on the CCS subsystem configuration API.
 * An instance will be the receptacle of all the subsystem configuration/persistent data which is declared
 * here using the {@link ConfigurationParameter} annotation.
 * @author tether
 */
public class CCSConfiguration implements LocalConfigService, HasLifecycle {

    @LookupField(strategy=Strategy.TREE)
    private volatile ConfigurationService ccsConfig;

    @LookupField(strategy=Strategy.TREE)
    private volatile AlertService ccsAlert;

    ////////// Implementation of LocalConfigService //////////
    private volatile Duration readoutInterval;
    private volatile List<InstrumentConfig> instConfigs;


    /**
     * Checks configuration and persistent data and makes the configuration objects.
     * Must only be called when full configuration data is available.
     */
    @Override
    public void makeConfigurationObjects() {
        try {
            readoutInterval = Duration.parse(readoutIntervalString);
        }
        catch (java.time.DateTimeException exc) {
            throw new IllegalArgumentException("Bad configuration. Invalid readout interval.");
        }
        makeInstrumentConfigs();
    }

    private void makeInstrumentConfigs() {
        // All the "instrumentXXX" configuration arrays must have the same length.
        final int numInstruments = this.instrumentLocations.length;
        if (
            this.instrumentConnections.length != numInstruments ||
            this.instrumentShortNames.length  != numInstruments ||
            this.instrumentTypes.length       != numInstruments || 
            this.instrumentWorking.length     != numInstruments
        )
        {
            throw new IllegalArgumentException(
                "Bad configuration. Not all the instrumentXxx arrays are the same length."
            );
        }
        if (numInstruments == 0) {
            throw new IllegalArgumentException("Bad configuration. No instruments defined!");
        }

        // Is every instrument type found in the InstrumentType enum?
        final List<InstrumentType> types = new ArrayList<>(numInstruments);
        for (final String configType: instrumentTypes) {
            final Optional<InstrumentType> enumType = InstrumentType.parse(configType);
            if (!enumType.isPresent()) {
                throw new IllegalArgumentException(String.format("\"%s\" is not a valid instrument type.", configType));
            }
            types.add(enumType.get());
        }

        // Construction of configuration objects.
        instConfigs = Collections.unmodifiableList(
            IntStream
            .range(0, numInstruments)
            .mapToObj(i -> new InstrumentConfig(
                    i,
                    types.get(i),
                    instrumentConnections[i],
                    instrumentLocations[i],
                    instrumentShortNames[i],
                    instrumentWorking[i] != 0,
                    this,
                    ccsAlert
            ))
            .collect(Collectors.toList()));
    }


    /** {@inheritDoc } */
    @Override
    public List<InstrumentConfig> getInstrumentConfigs() {
        return instConfigs;
    }
    
    /** {@inheritDoc } */
    @Override
    public List<String> getDewShortNames() {
        return Collections.unmodifiableList(Arrays.asList(dewShortNames));
    }

    /** {@inheritDoc } */
    @Override
    public Duration getReadoutInterval() {return readoutInterval;}
    
    /** {@inheritDoc} */
    @Override
    public String getEmailSender() {return emailSender;}
    
    /** {@inheritDoc } */
    @Override
    public String getEmailBounceAddress() {return emailBounceTo;}
    
    /** {@inheritDoc } */
    @Override
    public String getSMTPServer() {return SMTPServer;}
    
    /** {@inheritDoc } */
    @Override
    public List<String> getEmailRecipients() {
        return Collections.unmodifiableList(Arrays.asList(emailRecipients));
    }

    @Override
    /** {@inheritDoc} */
    public List<String> getOPCServerInfo(final String serverKey) {
        return Collections.unmodifiableList(this.opcServers.get(serverKey));
    }

    ////////// Lifecycle methods //////////
    /** Makes an instance. */
    public CCSConfiguration() {
        super();
    }

    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (ccsConfig == null) {
            missing.add("CCS configuration service");
        }
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
    }

    ////////// Configuration data //////////
    private static final int MAX_INSTRUMENTS = 10;

    /**
     * The time between readouts of all the instruments.
     */
    @ConfigurationParameter(name="readoutInterval", description="The time between readouts.")
    private volatile String readoutIntervalString = "PT1H";

    /**
     * For each instrument, its type. See {@link InstrumentType}.
     */
    @ConfigurationParameter(description="Instrument make and/or model.", maxLength=MAX_INSTRUMENTS)
    private volatile String[] instrumentTypes = new String[0];

    /**
     * For each instrument, a string containing connection information to be parsed.
     * For example "serial /dev/ttyUSB0" or "net foo.slac.stanford.edu 2250".
     * The nature of the information depends on the type of instrument. See the classes
     * which implement {@link Instrument}.
     */
    @ConfigurationParameter(description="How to open a connection to the instrument.", maxLength=MAX_INSTRUMENTS)
    private volatile String[] instrumentConnections = new String[0];

    /**
     * For each instrument, the "official" name of its location.
     */
    @ConfigurationParameter(description="Locations of instruments.", maxLength=MAX_INSTRUMENTS)
    private volatile String[] instrumentLocations = new String[0];
    
    /**
     * For each instrument, the "short name" or nickname for its location.
     */
    @ConfigurationParameter(description="Short names of instruments.", maxLength=MAX_INSTRUMENTS)
    private volatile String[] instrumentShortNames = new String[0];
    
    @ConfigurationParameter(description="Short names of instruments to use for dew point calaculation.", maxLength=MAX_INSTRUMENTS)
    private volatile String[] dewShortNames = new String[0];
    
    /**
     * For each instrument, a flag telling whether it's present and in working order.
     * Zero for a missing or broken instrument, anything else for working one.
     * This flag is used to set the initial value of the "enabled" flag
     * for each instrument.
     */
    @ConfigurationParameter(description="Which instruments are working? 0 = not working.", maxLength=MAX_INSTRUMENTS)
    private volatile int[] instrumentWorking = new int[0];
    
    /**
     * The name or IPv4 address of the SMTP server to use to send emails.
     */
    @ConfigurationParameter(description="The name or IPv4 address of the SMTP server.")
    private volatile String SMTPServer = "";
    
    /**
     * What to put in the From: fields of emails headers.
     */
    @ConfigurationParameter(description="Email address, possibly fake, used as email sender.")
    private volatile String emailSender = "";
    
    /**
     * What to put in the Bounce-to: fields for alarm emails. Must be a the address of
     * a real mailbox.
     */
    @ConfigurationParameter(description="Real email address which receives bounced messages.")
    private volatile String emailBounceTo = "";
    
    /**
     * The list of email recipients. A set of real email addresses.
     */
    @ConfigurationParameter(description="List of recipients of alarm emails.", maxLength=10)
    private volatile String[] emailRecipients = new String[] {};

    /**
     * A map where each value is a list of strings used to connect to an OPC server
     * publishing Lighthouse sensor data. The key either the Windows hostname
     * or some nickname for it.
     */
    @ConfigurationParameter(description="Connection information for LMS Express OPC servers.", maxLength=10)
    private volatile Map<String, List<String>> opcServers = Collections.emptyMap();
}
