package org.lsst.ccs.messaging.util;

import java.util.concurrent.locks.LockSupport;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * Rate limiter for outgoing messages.
 *
 * @author onoprien
 */
public class Throttle {

// -- Fields : -----------------------------------------------------------------
    
    static final long MIN_DELAY = 1000; // ns
    static private final long VETO_TIME = 60*1000000000; // 1 minute
    static private final double VETO_FACTOR = 2;
    
    private final Dispatcher dispatcher;
    private final Alert alert;
    
    private final long maxSize; // Bytes
    private final double maxRate; // Bytes/ns
    
    private long prevDelay, prevTime;
    
    private long accSize = 0;
    private long accTime = System.nanoTime();

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance, registering new {@code Alert} with the {@code Dispatcher}.
     * 
     * @param dispatcher Dispatcher through which alerts should be raised and messages logged.
     * @param alertID ID for {@code Alert} raised through this gate.
     * @param sizeMB Accumulated message size threshold.
     * @param rateMB Rate limit in MB/s
     */
    public Throttle(Dispatcher dispatcher, String alertID, int sizeMB, int rateMB) {
        
        maxSize = sizeMB * 1000000L;
        maxRate = rateMB / 1e3;
        
        alert = new Alert(alertID, "Message rate alert.");
//        dispatcher.registerAlert(alert);
        this.dispatcher = dispatcher;
    }
    
// -----------------------------------------------------------------------------
    
    /**
     * Processes the message, taking an appropriate action if rate exceeds threshold.
     * @param message Message. If {@code null}, do nothing.
     */
    public void process(BusMessage message) {
        long size = message.getSize();
        if (size <= 0) return;
        long now = System.nanoTime();
        accSize = Math.max(0L, Math.round(accSize - maxRate*(now-accTime)));
        accSize += size;
        accTime = now;
        if (accSize > maxSize) {
            long delay = Math.round((accSize - maxSize)/maxRate);
            if (delay > MIN_DELAY) {
                if (delay > prevDelay * VETO_FACTOR || now > prevTime + VETO_TIME) {
                    Level level = delay > 1000000000 ? Level.WARNING : Level.FINE; // warning if > 1s
                    dispatcher.getLogger().log(level, "Throttle: {0} {1} of size {2} is held for {3} ms.", new Object[]{alert.getAlertId(), message.getClass().getSimpleName(), size, delay / 1000000});
                    prevDelay = delay;
                    prevTime = now;
                }
                LockSupport.parkNanos(delay);
                now = System.nanoTime();
                accSize = Math.max(0L, Math.round(accSize - maxRate*(now-accTime)));
                accTime = now;
            }
        }
    }    

}
