package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.ServiceLoader;

/**
 * Factory for {@link BusMessagingLayer} instances.
 */
public abstract class TransportManager {
    
    static final ArrayList<TransportManager> transportManagers = new ArrayList<>() ;
    
    static {
        ServiceLoader<TransportManager> loader = ServiceLoader.load(TransportManager.class) ;
        for(TransportManager transportManager : loader) {
            transportManagers.add(transportManager) ;
        }
    }

    /**
     * Creates an instance of {@code BusMessagingLayer}.
     * 
     * @param protocolstring String that specifies messaging protocol and its configuration, in 
     *                       {@code protocol[:configuration_file_name[:additional_configuration]} format..
     * @return New instance of {@code BusMessagingLayer}.
     * @throws TransportException if anything goes wrong.
     */
    public static BusMessagingLayer getConnection(String protocolstring) 
        throws TransportException{
        for(TransportManager manager : transportManagers) {
            BusMessagingLayer transport = manager.getInstance(protocolstring) ;
            if(transport != null) {
                return transport ;
            }
        }
        throw new TransportException(protocolstring) ;
    }
    
    /**
     * Method to be implemented by concrete transport managers to create messaging layers.
     * 
     * @param protocolString String that specifies messaging protocol and its configuration, in 
     *                       {@code protocol[:configuration_file_name[:additional_configuration]} format..
     * @return New instance of {@code BusMessagingLayer}, or {@code null} if the specified
     *         {@code protocolstring} is not applicable to this {@code TransportManager}.
     */
    protected abstract BusMessagingLayer getInstance(String protocolString) ;
}
