package org.lsst.ccs.messaging.util;

import java.util.logging.Level;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;

/**
 * Raises an alert or logs a message when the input crosses thresholds, taking care 
 * not to do so too often for the same reason. This class is thread-safe.
 *
 * @author onoprien
 */
public class AlertGate {

// -- Fields : -----------------------------------------------------------------
    
    private final Dispatcher dispatcher;
    private final Alert alert;
    
    private final int vetoTime;
    private final double vetoFactor;
    private final int[] thresholds;
    
    private int lastValue;
    private long lastTime;
    private volatile int lastLevel = -1;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance, registering new {@code Alert} with the {@code Dispatcher}.
     * 
     * @param dispatcher Dispatcher through which alerts should be raised and messages logged.
     * @param alertID ID for {@code Alert} raised through this gate.
     * @param alertDescription Description of the {@code Alert}.
     * @param thresholds Array of thresholds. Contains no more than 3 elements: log, warning, alarm.
     * @param vetoTime Veto time in milliseconds.
     * @param vetoFactor Veto factor.
     */
    public AlertGate(Dispatcher dispatcher, 
                     String alertID, String alertDescription,
                     int[] thresholds,
                     int vetoTime, double vetoFactor) {
        
        if (thresholds == null || thresholds.length == 0) {
            this.alert = null;
            this.thresholds = null;
        } else {
            this.alert = new Alert(alertID, alertDescription);
            this.thresholds = thresholds;
            dispatcher.registerAlert(alert);
        }
        this.dispatcher = dispatcher;
        this.vetoTime = vetoTime;
        this.vetoFactor = vetoFactor;
    }
    
    /**
     * Creates an instance with {@code vetoTime = 5000} and {@code vetoFactor = 1.5}.
     * The new {@code Alert} is registered with the {@code Dispatcher}.
     * 
     * @param dispatcher Dispatcher through which alerts should be raised and messages logged.
     * @param alertID ID for {@code Alert} raised through this gate.
     * @param alertDescription Description of the {@code Alert}.
     * @param thresholds Array of thresholds.
     */
    public AlertGate(Dispatcher dispatcher, 
                     String alertID, String alertDescription,
                     int[] thresholds) {
        
        this(dispatcher, 
             alertID, alertDescription,
             thresholds,
             5000, 1.5);
    }
    
// -----------------------------------------------------------------------------
    
    /**
     * Checks the value and takes appropriate action.
     * 
     * @param value Current value of the watched quantity.
     */
    public void check(int value) {
        if (thresholds == null) return;
        int level = -1;
        for (int i = 0; i<thresholds.length; i++) {
            if (value > thresholds[i]) {
                level = i;
            } else {
                break;
            }
        }
        if (level == -1 && lastLevel == -1) { // just another normal value
            return;
        }
        synchronized (this) {
            if (level == -1) { // back to normal
                lastValue = Integer.MIN_VALUE;
                lastTime = 0;
                if (lastLevel > 0) {
                    dispatcher.raiseAlert(alert, AlertState.NOMINAL, getMessage(value));
                }
                lastLevel = -1;
            } else {
                long currentTime = System.currentTimeMillis();
                if (level != lastLevel || (currentTime - lastTime) > vetoTime || value > (lastValue * vetoFactor)) {
                    lastValue = value;
                    lastTime = currentTime;
                    String message = getMessage(value);
                    Level logLevel = level == 0 ? Level.INFO : (level == 1 ? Level.WARNING : Level.SEVERE);
                    dispatcher.getLogger().log(logLevel, message);
                    if (level != 0 || lastLevel != -1) {
                        dispatcher.raiseAlert(alert, AlertState.values()[level], getMessage(value));
                    }
                    lastLevel = level;
                }
            }
        }
    }
    
    protected String getMessage(int value) {
        return "High value of "+ alert.getAlertId() +" : "+ value;
    }

}
