package org.lsst.ccs.messaging.util;

import java.util.*;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.TransportStateException;

/**
 * Task dispatcher that connects messaging framework (currently, JGroups) to CCS framework.
 * A singleton {@code Dispatcher} is created by CCS {@code MessagingLayer}. A single {@code Dispatcher}
 * handles all three buses.
 * <p>
 * {@code MessagingLayer} submits executable tasks (incoming message processing or disconnection notifications) through
 * the {@link #in in(...)} method. CCS framework submits message sending tasks through the {@link #out out(...)} method.
 * <p>
 * {@code Dispatcher} also provides API for the messaging system to send events (statistics, alerts, etc.)
 * to other CCS components. Various CCS components can retrieve information about the messaging system
 * and its current state through {@code Dispatcher}.
 * <p>
 * All implementations should provide public constructor that takes {@code String} argument.
 * This is a way to supply configuration parameters through the protocol string passed to {@code TransportManager}.
 * The string should be in {@code "key[=value]&...&key[=value]"} format, where value may be a comma-separated array.
 * Prohibited characters: {@code :?}.
 * <p>
 * Implementations should be thread-safe, methods of this interface can be called on any thread.
 *
 * @author onoprien
 */
public interface Dispatcher {
    
    /** Enumeration of order of execution and capacity control policies. */
    public enum Order {
        /** Normal. */ NORM,
        /** Out-of-band, capacity-controlled. */ OOB_CC,
        /** Out-of-band, no capacity-controls. */ OOB}
    
    /** Enumeration of task execution stages for which statistical data can be reported. */
    public enum Stage {
        /** Execution of a call that submits a task for processing has started. Duration of this stage is always zero. */ START,
        /** Waiting in the {@code Dispatcher} queue. */ WAIT,
        /** Task execution. */ RUN,
        /** Between the start end the end of a call that submits a task for processing. This stage may overlap with others. */ SUBMIT
    }
    

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Initializes this dispatcher.
     */
    void initialize();
    
    /**
     * Orderly shuts down this dispatcher. Stops accepting tasks, executes previously
     * submitted tasks, delivers events to listeners, then shuts down.
     * The {@code MessagingLayer} will not be closed until this method returns.
     */
    void shutdown();
    

// -- Task submission : --------------------------------------------------------
    
    /**
     * Submits a task to process an incoming message or disconnection notification.
     * Depending on the implementation, this method may return immediately or after the task has been executed.
     * <p>
     * If one or more agent names are given, the task if guaranteed to be executed after any previously submitted
     * tasks for the same bus and agent. If no agents are specified, the task is independent of other tasks,
     * subject to capacity controls imposed by the implementation of this {@code Dispatcher}. If the {@code agents}
     * argument is {@code null}, the task is independent of other tasks, not subject to capacity controls.
     * <p>
     * Calling this method has no effect if this Dispatcher has not been initialized or it has been shut down.
     * No exceptions are thrown in this case.
     * 
     * @param task Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param agents Names of affected agents (source or disconnected).
     * @throws TransportStateException If this Dispatcher is unable to accept the
     *         request for reasons other than being uninitialized or shut down.
     */
    void in(Runnable task, Bus bus, String... agents);
    
    /**
     * Submits a task to process an outgoing message.
     * Depending on the implementation, this method may return immediately or after the task has been executed.
     * <p>
     * Tasks submitted with {@code outOfBand} equal to {@code false} for the same bus are guaranteed to be
     * processed in the order of submission. If {@code outOfBand} equals {@code true}, the task is executed 
     * independently of others, subject to capacity controls imposed by the implementation of this service.
     * If {@code outOfBand} equals {@code null}, the task is independent, not subject to capacity controls.
     * 
     * @param task Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param order Order of execution and capacity control policies.
     * @throws TransportStateException If this Dispatcher is unable to accept the request.
     */
    void out(Runnable task, Bus bus, Order order);


// -- Events and listeners : ---------------------------------------------------
    
    /**
     * An event emitted by {@link Dispatcher}.
     * Clients interested in these events should implement {@link Listener}.
     */
    static public class Event {
        
        protected final Dispatcher dispatcher;
        
        protected Event(Dispatcher dispatcher) {
            this.dispatcher = dispatcher;
        }

        public Dispatcher getDispatcher() {
            return dispatcher;
        }
        
    }
    
    /**
     * Interface to be implemented by classes that should be notified of {@link Dispatcher} events.
     * Events are guaranteed to be delivered in proper sequence to each listener. There is no
     * guarantee that the {@code Dispatcher} will remain in the state it was in when the event was
     * emitted, and there is no synchronization between delivery to different listeners.
     */
    public interface Listener {
        void onEvent(Event event);
    }
    
    /**
     * Registers a listener.
     * @param listener Listener to add.
     */
    void addListener(Listener listener);
    
    /**
     * Removes a listener.
     * @param listener Listener to remove.
     */
    void removeListener(Listener listener);
    
    /**
     * Notifies listeners of the given event.
     * This method is public to allow clients publish events through this {@code Dispatcher}.
     * @param event 
     */
    void fireEvent(Event event);
    
    
// -- Alerts and logging : -----------------------------------------------------
    
    /**
     * {@link Dispatcher} event that triggers an alert.
     * Dispatchers may emit these events when their internal becomes abnormal (long task
     * queues, delays, etc). External messaging code can also trigger these events by calling
     * one of the {@code raiseAlert(...)} methods.
     */
    static public class AlertEvent extends Event {
        
        private final Alert alert;
        private final AlertState severity;
        private final String cause;
        
        public AlertEvent(Dispatcher dispatcher, Alert alert, AlertState severity, String cause) {
            super(dispatcher);
            this.alert = alert;
            this.severity = severity;
            this.cause = cause;
        }

        public Alert getAlert() {
            return alert;
        }

        public AlertState getSeverity() {
            return severity;
        }

        public String getCause() {
            return cause;
        }
        
    }
    
    /**
     * Notifies listeners by emitting an {@code AlertEvent}.
     * If an alert with matching {@code id} has not been registered, nothing is published.
     * 
     * @param id Alert id.
     * @param severity Alert severity.
     * @param cause Message.
     */
    void raiseAlert(String id, AlertState severity, String cause);
    
    /**
     * Notifies listeners by emitting an {@code AlertEvent}.
     * An event is published even if an alert with matching {@code id} has not been registered.
     * 
     * @param alert Alert instance.
     * @param severity Alert severity.
     * @param cause Message.
     */
    void raiseAlert(Alert alert, AlertState severity, String cause);
    
    /**
     * Returns the list of alerts registered with this {@code Dispatcher}.
     * @return Registered alerts.
     */
    List<Alert> getRegisteredAlerts();
    
    /**
     * Registers an alert with this Dispatcher.
     * @param alert Alert to add.
     */
    void registerAlert(Alert alert);
    
    /**
     * Returns a logger to be used for Dispatcher related messages.
     * @return Logger associated with this dispatcher.
     */
    Logger getLogger();
    
    
// -- Instrumented task class : ------------------------------------------------
    
    /**
     * Instrumented {@code Runnable} for submission to {@link Dispatcher}.
     * If a task submitted to {@code Dispatcher} implements this interface, it
     * will receive callbacks at the end of each processing stage.
     */
    public interface Task extends Runnable {
                
        /**
         * Called whenever a stage has been completed.
         * @param stage Stage that have been just completed.
         */
        void stageEnded(Stage stage);
        
    }

}
