package org.lsst.ccs.messaging.util;

import java.lang.management.ManagementFactory;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Managed executor that times its tasks.
 *
 * @author onoprien
 */
public class TimedExecutor extends ThreadPoolExecutor implements TimedExecutorMBean {

// -- Fields : -----------------------------------------------------------------
    
    private final Logger logger;
    private final String name;
    
    volatile private int millisWaiting;
    volatile private int millisExecuting;

// -- Life cycle : -------------------------------------------------------------
    
    private TimedExecutor(String threadName, Logger logger) {
        super(1, 1, 0L, TimeUnit.SECONDS, new LinkedBlockingQueue<>());
        this.name = threadName;
        this.logger = logger;
        super.setThreadFactory(new ThreadFactory() {
            private final ThreadFactory delegate = Executors.defaultThreadFactory();
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = delegate.newThread(r);
                thread.setDaemon(true);
                thread.setName(name);
                thread.setUncaughtExceptionHandler((t, x) -> logger.warn(name + ": uncaught exception thrown by " + name, x));
                return thread;
            }
        });
    }
    
    static public TimedExecutor newInstance(String name, Logger logger) {
        TimedExecutor exec = new TimedExecutor(name, logger);
        try {
            ObjectName mbeanName = new ObjectName(name + ":type=TimedExecutor");
            ManagementFactory.getPlatformMBeanServer().registerMBean(exec, mbeanName);
        } catch (MalformedObjectNameException | InstanceAlreadyExistsException | MBeanRegistrationException | NotCompliantMBeanException x) {
            logger.error("Unable to register TimedExecutorMBean", x);
        }
        return exec;
    }

// -- Getters : ----------------------------------------------------------------
    
    @Override
    public String getName() {
        return name;
    }

    @Override
    public int getQueueSize() {
        return getQueue().size();
    }

    @Override
    public int getMillisWaiting() {
        return millisWaiting;
    }

    @Override
    public int getMillisExecuting() {
        return millisExecuting;
    }

// -- Overriding behavior : ----------------------------------------------------

    @Override
    public void execute(Runnable r) {
        Task task = new Task(r);
        super.execute(task);
    }

    @Override
    protected void beforeExecute(Thread t, Runnable r) {
        Task task = (Task) r;
        long currentTime = System.currentTimeMillis();
        millisWaiting = (int) (currentTime - task.getTime());
        task.setTime(currentTime);
    }

    @Override
    protected void afterExecute(Runnable r, Throwable t) {
        Task task = (Task) r;
        millisExecuting = (int) (System.currentTimeMillis() - task.getTime());
    }

    @Override
    protected void terminated() {
        try {
            ObjectName mbeanName = new ObjectName(name + ":type=TimedExecutor");
            ManagementFactory.getPlatformMBeanServer().unregisterMBean(mbeanName);
        } catch (MalformedObjectNameException | MBeanRegistrationException | InstanceNotFoundException x) {
            logger.error("Unable to register TimedExecutorMBean", x);
        }
        super.terminated();
    }
    
    
// -- Task class : -------------------------------------------------------------
    
    static private class Task implements Runnable {
        
        private final Runnable realRunnable;
        private long time = System.currentTimeMillis();
        
        Task(Runnable runnable) {
            realRunnable = runnable;
        }

        @Override
        public void run() {
            realRunnable.run();
        }

        public long getTime() {
            return time;
        }

        public void setTime(long time) {
            this.time = time;
        }
        
    }

}
