package org.lsst.ccs.messaging;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * Entry point for receiving messages from the buses.
 * Its name is used as a unique identifier on the buses.
 * @author emarin
 */
public class MessagingAccessLayer {
    
    private final AgentInfo agent;
    private final EnumMap<Bus, BusAccess> busAccesses = new EnumMap<>(Bus.class);

    public MessagingAccessLayer(AgentInfo agent, BusAccess... buses){
        this.agent = agent;
        for (BusAccess busAccess : buses) {
            busAccesses.put(busAccess.getBus(), busAccess);
        }
    }

    /**
     * Returns the name of the associated {@code Agent}.
     * This name must be unique on the buses.
     * The uniqueness is enforced when connecting with the messaging layer.
     * @return Agent name. 
     */
    public String getName(){
        return agent.getName();
    }
    
    /**
     * Returns the descriptor of the associated {@code Agent}.
     * @return Agent descriptor.
     */
    public AgentInfo getAgentInfo() {
        return agent;
    }
    
    /**
     * This list of buses might be used by the messaging layer to determine on which buses
     * the associated agent has to be connected to.
     * @return The list of Buses the layer has to be registered on.
    */
    public Collection<BusAccess> getBusAccesses(){
        return busAccesses.values();
    }
    
    public BusAccess getBusAccess(Bus bus){
        return busAccesses.get(bus);
    }
    
    /**
     * Add a BusMessagePreProcessor the the MessagingAccessLayer.
     * The ButMessagePreProcessor is added here to the appropriate Bus.
     * 
     * @param preProcessor The BysMessagePreProcessor to be added.
     */
    public void addBusMessagePreProcessor(BusMessagePreProcessor preProcessor) {
        busAccesses.get(preProcessor.getBus()).addPreProcessor(preProcessor);
    }
    
    /**
     * Handler for processing received messages.
     * Whenever a message is received, {@code update(...)} methods of all forwarders
     * listed by {@code getForwarderList()} are called, then {@code processBusMessage(...)} is called.
     * 
     * @param <T> the type of the received message
     */
    public static class BusAccess<T extends BusMessage> {

        // The forwarder map must preserve insertion order to ensure the 
        // agent presence manager will be the first one notified for status events
        private final Map<MessageListener, BusMessageForwarder> forwarderMap = 
                Collections.synchronizedMap(new LinkedHashMap<>());
        private final Bus bus;
        private final List<Predicate<BusMessage<? extends Serializable, ?>>> filters = new CopyOnWriteArrayList<>();
        private final List<BusMessagePreProcessor> preProcessors = new ArrayList<>();
        
        public BusAccess(Bus bus) {
            this.bus = bus;
        }
        
        public Bus getBus(){
            return bus;
        }
        
        public void processBusMessage(T message){
            //First reject messages that are not accepted because they
            //belong to different groups
            if ( ! acceptBusMessage(message) ) {
                return;
            }

            boolean forwardMessage = true;
            T originalMessage = message;
            for (BusMessagePreProcessor preProcessor : preProcessors) {
                T processedMessage = (T)preProcessor.preProcessMessage(message);
                if ( processedMessage == null ) {
                    forwardMessage = false;
                } else {
                    message = processedMessage;
                }
            }
            if ( ! forwardMessage ) {
                return;
            }
            //If we were given a completely different object we need to:
            //   - update the timing information: is this really needed?
            //   - update the message origin
            if ( message != originalMessage ) {
                message.updateTimingInformation(originalMessage);
                message.setOriginAgentInfo(originalMessage.getOriginAgentInfo());
            }
            
                        
            Collection<BusMessageForwarder> c;
            synchronized(forwarderMap) {
                c = new ArrayList<>(forwarderMap.values());
            }
            for(BusMessageForwarder bmf : c) {
                bmf.update(message);
            }
        }
        
        public void addPreProcessor(BusMessagePreProcessor preProcessor) {
            preProcessors.add(preProcessor);
        }
        
        public void addForwarder(MessageListener l, BusMessageForwarder forwarder){
            forwarderMap.put(l, forwarder);
        }
        
        public void removeForwarder(MessageListener l){
            forwarderMap.remove(l);
        }

        public void processClusterDeserializationError(String address, RuntimeException e) {

        }
        
        public void addBusMessageFilter(Predicate<BusMessage<? extends Serializable, ?>> filter) {
            filters.add(filter);
        } 

        private boolean acceptBusMessage(BusMessage msg) {
            if ( filters.isEmpty() ) {
                return true;
            }
            for (Predicate<BusMessage<? extends Serializable, ?>> filter : filters) {
                boolean accept = filter.test(msg);
                if (!accept) {
                    return false;
                }
            }
            return true;
        }        
        
    }
    
}
