package org.lsst.ccs.messaging;

import java.util.Comparator;
import java.util.SortedSet;
import java.util.Spliterator;
import java.util.TreeSet;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.lsst.ccs.bus.messages.LogMessage;

public class LogAggregator implements LogMessageListener, Spliterator<LogMessage> {

	long windowWidth = 20; // in ms, to order messages by originating timestamp

	SortedSet<LogMessage> messages = new TreeSet<LogMessage>(logComparator);

	// Note: this comparator imposes orderings that are inconsistent with
	// equals.
	static class LogComparator implements Comparator<LogMessage> {

		@Override
		public int compare(LogMessage m1, LogMessage m2) {
                    return m1.getCCSTimeStamp().getUTCInstant().compareTo(m2.getCCSTimeStamp().getUTCInstant());
		}
	}

	static LogComparator logComparator = new LogComparator();

	@Override
	public void onLogMessage(LogMessage msg) {
		push(msg);
	}

	synchronized void push(LogMessage msg) {
		messages.add(msg);
		notify();
	}

	synchronized LogMessage pull() {
		LogMessage m;
		while (messages.isEmpty() || (m = messages.first()).getCCSTimeStamp().getUTCInstant().toEpochMilli() > System.currentTimeMillis() - windowWidth) {
			try {
				wait(messages.isEmpty() ? 100 : windowWidth);
			} catch (InterruptedException e) {
				// ignore
			}
		}
		messages.remove(m);
		return m;
	}

	@Override
	public boolean tryAdvance(Consumer<? super LogMessage> action) {
		action.accept(pull());
		return true;
	}

	@Override
	public Spliterator<LogMessage> trySplit() {
		return null;
	}

	@Override
	public long estimateSize() {
		return Long.MAX_VALUE;
	}

	@Override
	public int characteristics() {
		return DISTINCT | NONNULL | ORDERED | SORTED | CONCURRENT;
	}

	public Comparator<? super LogMessage> getComparator() {
		return logComparator;
	}

	public Stream<LogMessage> stream() {
		return StreamSupport.stream(this, false);
	}

}
