package org.lsst.ccs.messaging;

import org.lsst.ccs.bus.data.AgentInfo;

/**
 * Agents that implement this interface are notified when an agent is connecting
 * or has disconnected on the status bus
 * 
 * @author emarin
 */
public interface AgentPresenceListener {
    
    /**
     * Indicates that the list of agents represented by {@code agents} is present on the
     * buses.
     * 
     * This method is invoked when the listener is added with all the agents that
     * are already connected.
     * 
     * At this point not all services on the connecting agents 
     * are guaranteed to have completely started. This method should be used only
     * to receive a notification that an agent is connecting.
     * For a guarantee that an agent information is fully present, please use the
     * {@code #connected} methods. 
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * 
     * @param agents
     */
    public default void connecting(AgentInfo... agents) {

    }

    /**
     * Indicates that the agent represented by {@code agent} has left the buses.
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * @param agent 
     */
    public default void disconnecting(AgentInfo agent) {
        
    }
    
    
    /**
     * All the provided {@code agents} are fully connected to the buses, they
     * are OPERATIONAL and ready to receive commands.
     * 
     * This method is invoked when the listener is added with all the agents that
     * are already connected.
     * 
     * When this method is invoked the agent is in PhaseState::OPERATIONAL state
     * and all its services are guaranteed to have been properly initialized.
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * 
     * @param agents
     */
    public default void connected(AgentInfo... agents) {
    }
    
    
}
