package org.lsst.ccs.bus;

import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.tracers.Tracer;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.lsst.ccs.utilities.tracers.Names;

/**
 * Messaging factory, insulating the application from the messaging middleware
 * (JMS or other)
 *
 * @author aubourg
 */

public class MessagingFactory {
    static {
        assert Tracer.version("$Rev$", MessagingFactory.class, "org-lsst-ccs-messaging-layer") ;
    }
    protected volatile static MessagingFactory instance;
    private final Map<String,MessagingFactory.InnerFactory> map = new HashMap<>() ;
    private final Logger curLogger = Logger.getLogger("org.lsst.ccs.bus") ;
    private final Map<BusMessagingLayer, AgentPresenceManager> apmMap = new HashMap<>();

    /**
     * this boolean is used for tests where different subsystems
     * share the same JVM
     */
    static class BooleanWrapper {
        AtomicBoolean  atomicBoolean = new AtomicBoolean() ;
        public BooleanWrapper() {
            System.out.println(" build invoke " + this);
        }

        public boolean get() {
            return atomicBoolean.get();
        }

        public void set(boolean newValue) {
            atomicBoolean.set(newValue);
        }

        @Override
        public String toString() {
           return super.toString() +
                   " " + atomicBoolean ;
        }
    }

    //@TODO: should be getInstance( String subsystemName)
    public static synchronized MessagingFactory getInstance() {
        if (instance == null) {
            instanciate();
        }
        return instance;
    }
    
    protected static void instanciate() {
        try {
            instance = new MessagingFactory();
        } catch (Exception e) { //@modified (bamade) : for other crashes
            throw new RuntimeException(e);
        }
    }
    
    public MessagingFactory() {
    }
    
    // TODO : This field is never set, never accessed
    private String subsystemName;
    
    public  MessagingFactory forSubsystem(String name) {
        BusMessagingLayer busMessagingLayer ;
        String protocolProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport", "jgroups:udp_ccs:") ;
        String transportPropsProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport.properties", "");
        try {
            //busMessagingLayer = (BusMessagingLayer) Class.forName(clazzName).newInstance();
            busMessagingLayer = TransportManager.getConnection(protocolProperty,transportPropsProperty) ;
        } catch (TransportException exc) {
            //todo: change this exception
            throw new RuntimeException(exc) ;
        }
        /**/
        boolean toshareJVM = ("TRUE".equals(BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.sharedJVM", "false").toUpperCase()));

        MessagingFactory.InnerFactory fact = map.get(name) ;
        
        if(fact == null ) {
            fact = new MessagingFactory.InnerFactory(name, busMessagingLayer);
            map.put(name, fact) ;
        } else {
            //TODO  : get rid of this FUNKY CODE !!!!!
            /*
            when changing from a comm layer to another there was a bug (it kept the same layer for the same subsystem during tests)
            but the code should be completely removed when we get rid of Singletons!
            */
            BusMessagingLayer layer = fact.layer.getBusMessagingLayer() ;
            if(! layer.getClass().isAssignableFrom(busMessagingLayer.getClass())) {
                //System.out.println("OLD LAYER :" + layer + " NEW LAYER : " + busMessagingLayer);
                curLogger.warn("LAYER CHANGE! OLD LAYER :" + layer +
                        " NEW LAYER : " + busMessagingLayer);
                fact = new InnerFactory(name, busMessagingLayer);
                map.put(name, fact) ;
            }
        }
        fact.sharedJVM = toshareJVM ;
        return fact ;
        //	this.subsystemName=subsystemName;
        //      return this ;
    }
    
    public String getSubsystemName() {
        return subsystemName;
    }
    
    
    // TODO sendCommandSync qui retourne un Reply
    
    
    public boolean isReplyRequested() {
        return false;
    }
    
    //@modified (bamade) -> @TODO: add a close() method to stop listening threads!
    public void shutdownBusAccess() {
        
    }
    public List<String> connectedToCommand() {
        return Collections.EMPTY_LIST ;
    }
    
    private class InnerFactory extends MessagingFactory {

        final BusMessagingLayer transportLayer;
        final BusApplicationLayer layer ;
        private final boolean[] registered = new boolean[Bus.values().length];
        private boolean autoReply = true ;
        volatile boolean sharedJVM = false ;

        InnerFactory(String subSystem, BusMessagingLayer transport) {
            this.transportLayer = transport;
            this.layer = new BusApplicationLayer(subSystem, transport);
        }
        
        @Override
        public synchronized void addCommandListener(CommandListener l) {
            if(sharedJVM || ! registered[Bus.COMMAND.ordinal()]){
                try {
                    layer.registerToCommand();
                } catch(RuntimeException exc) {
                    throw exc ;
                } catch (Exception e) {
                    //TODO: change that!
                    throw new RuntimeException(e) ;
                }
                registered[Bus.COMMAND.ordinal()] = true ;
            }
            layer.addCommandListener(l);
        }
        
        @Override
        public synchronized void removeCommandListener(CommandListener l) {
            layer.removeCommandListener(l);
        }
        
        @Override
        public  synchronized void addStatusListener(StatusListens l) {
            if(! registered[Bus.STATUS.ordinal()]){
                try {
                    layer.registerToStatus();
                } catch (IOException e) {
                    throw new RuntimeException(e) ;
                }
                registered[Bus.STATUS.ordinal()] = true ;
            }
            layer.addStatusListener(l);
        }
        
        @Override
        public synchronized void removeStatusListener(StatusListens l) {
            layer.removeStatusListener(l);
        }
        
        @Override
        public synchronized void addLogListener(LogListener l) {
            if(! registered[Bus.LOG.ordinal()]){
                try {
                    layer.registerToLog();
                } catch (IOException e) {
                    throw new RuntimeException(e) ;
                }
                registered[Bus.LOG.ordinal()] = true ;
            }
            layer.addLogListener(l);
        }
        
        @Override
        public synchronized void removeLogListener(LogListener l) {
            layer.removeLogListener(l);
        }
        
        @Override
        public void addCommandListener(CommandListener l, String selector) {
            addCommandListener(l);
        }
        
        @Override
        public void addStatusListener(StatusListens l, String selector) {
            addStatusListener(l);
        }
        
        @Override
        public void addLogListener(LogListener l, String selector) {
            addLogListener(l);
        }
        
        @Override
        public synchronized void sendCommand(Command cmd) {
            try {
                if(! registered[Bus.COMMAND.ordinal()]){
                    layer.registerToCommand();
                    registered[Bus.COMMAND.ordinal()] = true ;
                }
                //TODO: check if Command without Origin!
                layer.sendCommand(cmd);
            } catch (IOException e) {
                //TODO:change that!!!
                if(e instanceof DestinationsException) {
                    //System.out.println(" Log WARN: " + e);
                    curLogger.warn("destination problem", e) ;
                    // toDO : wit until lock arbitrator
                    // The exception is thrown again to address https://jira.slac.stanford.edu/browse/LSSTCCS-181
                    // In the future we might want to have smarter logic to address any
                    // LockArbitrator issues.
                    throw new RuntimeException(e);
                } else {
                    throw new RuntimeException(e);
                }
            }
        }
        
        @Override
        public synchronized void sendStatus(Status status) {
            try {
                if(! registered[Bus.STATUS.ordinal()]){
                    layer.registerToStatus();
                    registered[Bus.STATUS.ordinal()] = true ;
                }
                layer.sendStatus(status);
            } catch (IOException e) {
                //TODO:change that!!!
                throw new RuntimeException(e);
            }
            //To change body of implemented methods use File | Settings | File Templates.
        }
        
        @Override
        public synchronized void sendLogEvent(LogEvent evt) {
            try {
                if(! registered[Bus.LOG.ordinal()]){
                    layer.registerToLog();
                    registered[Bus.LOG.ordinal()] = true ;
                }
                layer.sendLogEvent(evt);
            } catch (IOException e) {
                //TODO:change that!!!
                throw new RuntimeException(e);
            }
        }
        
        @Override
        public String getToken() {
            return layer.getToken();
        }
        
        @Override
        public void reply(CommandAckOrReply cmd) {
            try {
                layer.reply(cmd);
            } catch (IOException e) {
                //TODO:change that!!!
                throw new RuntimeException(e);
            }
        }
        
        @Override
        public void noAutoReply() {
            autoReply = false ;
        }
        
        // TODO: change that .....
        
        @Override
        public boolean isReplyRequested(){
            return this.autoReply ;
        }
        
        @Override
        public void shutdownBusAccess() {
            layer.close() ;
            for(int ix = 0 ; ix < registered.length; ix++) {
                registered[ix] = false ;
            }
        }
        
        @Override
        public List<String> connectedToCommand() {
            return layer.connectedToCommand() ;
        }
        
        @Override
        public synchronized void addMembershipListener(BusMembershipListener l, Bus bus) {
            layer.addMembershipListener(l, bus);
        }
        
        @Override
        public synchronized void removeMembershipListener(BusMembershipListener l, Bus bus) {
            layer.removeMembershipListener(l, bus);
        }
        
        @Override
        public synchronized AgentPresenceManager getAgentPresenceManager(){
            AgentPresenceManager agentPresenceManager = apmMap.get(transportLayer);
            if (agentPresenceManager == null){
                String apmName = Names.almostUniqueAgentName("apm");
                if (transportLayer instanceof ProvidesDisconnectionInformation){
                    agentPresenceManager = new AgentPresenceManagerBML();
                    layer.addMembershipListener((BusMembershipListener)agentPresenceManager, Bus.STATUS);
                } else {
                    agentPresenceManager = new DefaultAgentPresenceManager();
                }
                
                MessagingFactory.getInstance().forSubsystem(Names.almostUniqueAgentName(apmName)).addStatusListener(agentPresenceManager);
                apmMap.put(transportLayer,agentPresenceManager);
            }
            return agentPresenceManager;
        }
    }
    
    public void addCommandListener(CommandListener l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    public synchronized void removeCommandListener(CommandListener l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void addStatusListener(StatusListens l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    public synchronized void removeStatusListener(StatusListens l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void addLogListener(LogListener l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public synchronized void removeLogListener(LogListener l) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void addCommandListener(CommandListener l, String selector) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void addStatusListener(StatusListens l, String selector) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)");
    }
    
    
    public void addLogListener(LogListener l, String selector) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)");
    }
    
    
    public void sendCommand(Command cmd) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void sendStatus(Status status) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void sendLogEvent(LogEvent evt) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public String getToken() {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
        //return null;
    }
    
    
    public void reply(CommandAckOrReply cmd) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void noAutoReply() {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void addMembershipListener(BusMembershipListener l, Bus bus) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    
    public void removeMembershipListener(BusMembershipListener l, Bus bus) {
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
    public AgentPresenceManager getAgentPresenceManager(){
        throw new UnsupportedOperationException("incorrect call on factory: use forSubsystem(name)") ;
    }
    
}
