package org.lsst.ccs.bus;

import java.io.Closeable;
import java.io.IOException;

/**
  * Interface defining the access to the Transport Layer of Buses
  * @author bamade
 * TODO: create an accept URI
 *
  */
public interface BusMessagingLayer extends Closeable {
    public static final String ANONYMOUS_AGENT = "__" ;
    /**
     * Creates  low level communication entry points for a subsystem.
     * Abstract vision is that there is a different entry point for each bus/topic.
     *
     * @param agentName name of the sending/receiving point as will be known by transport
     * null or empty string  means that the local agent will receive all messages
     * (whatever the destination of the message is  : "anonymous agent")
     * if an agent with same name is already locally registered for these buses nothing happens (the call is
     * idempotent), but if another agent has the same name on the network a <TT>DuplicateBusNameException</TT>
     * may be fired (this is an optional behaviour) but this exception is reported only to the corresponding
     * <TT>BusMembershipListener</TT>
     *
     * @param buses list of  buses we want to connect to, if empty connects to all buses
     * @throws IOException if connection impossible,
     */
    public void register(String agentName, Bus... buses) throws IOException;

    /**
     * close  entry points for  a bus for an agent.
     * calls should be idempotent.
     * @param agentName (if empty voids the "anonymous" agent capabilities
     * @param buses if empty all registered buses for the agent will be closed
     * @throws IllegalArgumentException if you do not "own" the agent corresponding to the name.
     */
    public void closeFor(String agentName, Bus... buses);

    /**
     * Close the Transport Layer. Subsequent method calls will have no effect.
     * @throws IOException
     */
    @Override
    public void close() throws IOException ;

    /**
     * sends a message on a bus: note that BusMessage should have sender and
     * destination information but it is not the role of the communication layer
     * to parse destination information such as "subsystem1, subsystem2" or
     * "subsystem3/module"a.
     * <P>
     *     The message will be sent to all destinations plus to all the "anonymous" agents
     * </P>
     *
     * @param senderAgent which agent is supposed to be the initiator of the message
     * (anonymous agents are not supposed to send message: that may fire an exception)
     * @param bus
     * @param message
     * @param destinations could be empty (means broadcast) of a single "" or a single"*" (again means broadcast)
     * or a list of destinations (DO NOT use things such as "*", "dest1", "dest2" : this will not work!)
     * @throws IOException this could have a list of causes if some destinations fail.
     * as much as possible implementers will ensure that all correct destinationa are adressed:
     * if some fail it is not mandatory to report with a special subclass of IOexception that lists all destination
     * that failed.
     * @see org.lsst.ccs.bus.DestinationsException
     */
    public <T extends BusPayload> void sendMessage(String senderAgent, Bus<T> bus, T message,String... destinations) throws IOException ;

    /**
     * Sets up callback configuration for a topic and subsystem.
     * There could be multiple <TT>Forwarder</TT> for a given bus.
     * <B>All callbacks are supposed to be  multithreaded</B>
     * @param agentName if empty adds a forwarder to "anonymous" agent that receives all messages
     * @param forwarder code that handles the incoming messages
     * @param buses if empty the forwarder listens to all buses.
     * @throws IllegalArgumentException if the subsystem is not registered to one of the buses
     * or if forwarder is null
     */
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) ;

    /**
     * Removes a message listener from the forwarder list.
     * @param agentName
     * @param forwarder
     * @param buses if empty forwarder is removed from all buses
     * @throws IllegalArgumentException   if forwarder is null
     * (but no exception if the subsystem is not registered, or forwarder not present)
     */
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) ;

    /**
     * optional operation.
     * registers a listener for knowing about the agents that connect/disconnect on a bus.
     * This method is more useful if called before any agent registration (for instance
     * duplicate agent names will be reported to these listeners)
     *
     * @param buses if empty  register to all buses
     * @param listener can be null if we want to deregister a previous listener
     * @throws UnsupportedOperationException if not supported
     */
    public void setMembershipListener( BusMembershipListener listener, Bus... buses);

}
