package org.lsst.ccs.messaging.jms;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.BusMessageForwarder;
import org.lsst.ccs.messaging.BusMessagingLayer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.jms.InvalidClientIDException;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.messaging.DuplicateAgentNameException;
import org.lsst.ccs.messaging.MessagingAccessLayer;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.utilities.logging.Logger;

/**
 */
public class JMSBusMessagingLayer implements BusMessagingLayer {

    private static Logger logger = Logger.getLogger("org.lsst.ccs.bus.jms.JMSBusMessagingLayer");

    private Map<String, LocalAgent> mapLocalAgent = new ConcurrentHashMap<>();

    @Override
    public void connect(MessagingAccessLayer layer) throws DuplicateAgentNameException {
        String agentName = layer.getName();
        
        List<BusAccess> busAccesses = new ArrayList(layer.getBusAccesses());
        
        StringBuilder allBuses = new StringBuilder();
        for (BusAccess busAccess : busAccesses) {
            allBuses.append(busAccess.getBus().toString()).append(" ");
        }
        logger.fine("### Registering " + agentName + " for buses " + allBuses);

        // Is this agent known to the map ?
        LocalAgent localAgent = mapLocalAgent.get(agentName);
        if (localAgent == null) {
            localAgent = new LocalAgent();
            localAgent.agentName = agentName;
            mapLocalAgent.put(agentName, localAgent);
            logger.debug("context created");
            try {
                localAgent.session = TopicContextFactory.getTopicContextFactory().createSession(agentName);
            } catch (JMSException ex){
                if (ex instanceof InvalidClientIDException){
                    throw new DuplicateAgentNameException(agentName,"Channel with same name already exists");
                }
                else throw new RuntimeException(ex);
            }
        }

        for (BusAccess busAccess : busAccesses) {
            int index = busAccess.getBus().ordinal();

            localAgent.subscribers[index] = TopicContextFactory.getTopicContextFactory().getSubscriber(localAgent.session, agentName,busAccess, null);
           
            try {
                localAgent.receivers[index] = new BusReceiver(agentName, busAccess, localAgent.subscribers[index]);
            } catch (JMSException ex) {
                logger.error(JMSBusMessagingLayer.class.getName());
            }
        }
    }

    @Override
    public void disconnect(MessagingAccessLayer layer) {
        try {
            close();
        } catch (IOException ex) {
         logger.error(ex);
        }
    }

    class LocalAgent {

        String agentName;
        Session session;
        MessageConsumer[] subscribers = new MessageConsumer[Bus.values().length];
        BusReceiver[] receivers = new BusReceiver[Bus.values().length];
    }

    class BusReceiver implements MessageListener {

        BusAccess busAccess;
        //todo: duplicate information
        MessageConsumer curConsumer;
        //ParallelCommandDispatcher dispatcher;
        //todo: again a duplicate!
        String agentName;

        BusReceiver(String agentName, BusAccess busAccess, MessageConsumer curConsumer) throws JMSException {
            this.agentName = agentName;
            this.busAccess = busAccess;
            this.curConsumer = curConsumer;
            curConsumer.setMessageListener(this);
        }

        @Override
        public void onMessage(Message message) {

            BusMessage busMessInit = null;
            try {
//                if (busAccess.getForwarderList().isEmpty()) {
//                    return;
//                }

                try {
                    busMessInit = (BusMessage) ((ObjectMessage) message).getObject();
                } catch (JMSException ex) {
                    logger.error(JMSBusMessagingLayer.class.getName());
                }
            } catch (final RuntimeException exc) {
                throw exc;
                //TODO: create a generic Message for reporting errors
                /* busMessInit = new BusMessage() {
                 @Override
                 public String getMessageType() {
                 return "error" + exc;
                 }
                 } ;
                 */
            }
            final BusMessage busMess = busMessInit;
            busAccess.processBusMessage(busMess);
        }
    }

    // Close the entire local connections
    @Override
    public void close() throws IOException {
        Iterator<String> itKeys = mapLocalAgent.keySet().iterator();
        while (itKeys.hasNext()) {
            String key = itKeys.next();
            logger.fine("******** removing " + key);
            LocalAgent localAgent = mapLocalAgent.get(key);
            for (MessageConsumer mc : localAgent.subscribers) {
                if (mc != null) {
                    try {
                        mc.close();
                    } catch (JMSException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            }
            //updateConnectionsMap();
            // TODO : Do we need to destroy the connection itself ?
        }
    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }

        for (Bus bus : buses) {
            BusReceiver receiver = mapLocalAgent.get(agentName).receivers[bus.ordinal()];
            if (receiver == null) {
                throw new IllegalArgumentException(" agent " + agentName + "not registered on bus " + bus);
            }
     //       receiver.addForwarder(forwarder);
        }
    }

    @Override
    public Set<String> getRegisteredLocalAgents(Bus... buses) {
        // If there are no provided buses, it means for all buses.
        if (buses == null || buses.length == 0) {
            buses = Bus.values();
        }
        Set<String> result = new HashSet<>();

        for (String agent : mapLocalAgent.keySet()) {

            LocalAgent localAgent = mapLocalAgent.get(agent);

            for (Bus bus : buses) {
                int index = bus.ordinal();
                if (localAgent.subscribers[index] != null) {
                    if (!result.contains(agent)) {
                        result.add(agent);
                    }
                }
            }
        }
        return result;
    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgent localAgent = mapLocalAgent.get(agentName);
        if (localAgent != null) {
            for (Bus b : buses) {
                int index = b.ordinal();
                try {
                    localAgent.subscribers[index].close();
                    localAgent.session.unsubscribe(agentName);
                } catch (JMSException ex) {
                 throw new RuntimeException(ex);
                }
            }
        }
        // updateConnectionsMap();
    }

    @Override
    synchronized public <T extends BusMessage> void sendMessage(String senderAgent, Bus bus, T message) {
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        int index = bus.ordinal();

        Session s = mapLocalAgent.get(senderAgent).session;
        MessageProducer send = null;
        try {
            send = s.createProducer(TopicContextFactory.getTopicContextFactory().getTopic(bus.toString()));
        } catch (JMSException ex) {
            throw new RuntimeException(ex);
        }

        sendMessageToDestination(s, message, "", send, bus);
    }
    
    public <T extends BusMessage> void sendMessageToDestination(Session s, T message, String dest, MessageProducer send, Bus bus) {
        ObjectMessage m;
        try {
            m = s.createObjectMessage();
            m.setObject(message);
            //m.setJMSType(message.getMessageType());
            if (!dest.isEmpty()) {
                m.setStringProperty("destination", dest);
            }
            //m.setJMSPriority(convertToJMSPriority(message.getPriorityLevel()));
            //if (message.getCorrelId() != null) {
            //    m.setJMSCorrelationID(cmd.getCorrelId());
            //}

        } catch (JMSException e) {
            logger.error(e);
            throw new RuntimeException(e);
        }
        try {
            // if (m.getJMSCorrelationID() == null) {
            //     m.setJMSCorrelationID(localCorrelID.get());
            // }
            m.setJMSMessageID("0"); /* TODO handle messageid */

            m.setJMSPriority(4); // 0-4 normal, 4-9 expedited

            // uncomment to send replies on the command topic
            m.setJMSReplyTo(TopicContextFactory.getTopicContextFactory().getCommandTopic());

            m.setJMSTimestamp(System.currentTimeMillis());
            String busName = bus.toString();

            logger.debug("sending message " + m.getObject());
            send.send(m);
            //   send.
        } catch (JMSException e) {
            logger.error("Error sending message", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgent localAgent = mapLocalAgent.get(agentName);
        if (localAgent == null) {
            return;
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgent.receivers[bus.ordinal()];
            if (receiver == null) {
                continue;
            }
         //   receiver.removeForwarder(forwarder);
        }
    }

    @Override
    public void register(String agentName, Bus... buses) throws IOException {

       
    }
}
