package org.lsst.ccs.messaging.jgroups;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.function.Supplier;
import org.jgroups.Address;
import org.jgroups.conf.ClassConfigurator;
import org.jgroups.stack.AddressGenerator;
import org.jgroups.util.Bits;
import org.jgroups.util.UUID;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * {@code Address} that contains integer rank.
 * Ranks are used when selecting cluster coordinator.
 *
 * @author onoprien
 */
public class CCSAddress extends UUID {

// -- Fields : -----------------------------------------------------------------
    
    static final int RANK_RANGE = AgentInfo.AgentType.values().length;
    private long rank;


// -- Life cycle : -------------------------------------------------------------
    
    static {
        ClassConfigurator.add((short)7900, CCSAddress.class);
    }
    
    public CCSAddress() {
    }
    
    public CCSAddress(long rank) {
        super(generateRandomBytes());
        this.rank = rank;
    }

    @Override
    public Supplier<? extends CCSAddress> create() {
        return CCSAddress::new;
    }
    

// -- Getters : ----------------------------------------------------------------
    
    public long getRank() {
        return rank;
    }
    
    
// -- JGroups4 serialization : -------------------------------------------------

    @Override
    public void writeTo(DataOutput out) throws IOException {
        super.writeTo(out);
        Bits.writeLongCompressed(rank, out);
    }

    @Override
    public void readFrom(DataInput in) throws IOException {
        super.readFrom(in);
        rank = Bits.readLongCompressed(in);
    }

    @Override
    public int serializedSize() {
        return super.serializedSize() + Bits.size(rank);
    }
    
    
// -- Generator class : --------------------------------------------------------
    
    static public class Generator implements AddressGenerator {
        
        private final long rank;
        
        public Generator(AgentInfo agent) {
            long type = RANK_RANGE - agent.getType().ordinal() - 1;
            type <<= 59;
            long time = agent.getAgentStartTime().getUTCInstant().toEpochMilli();
            time &= 0x07_FF_FF_FFL;
            rank = type | time;
        }
        
        public Generator(long type) {
            type = RANK_RANGE - type - 1;
            type <<= 59;
            long time = System.currentTimeMillis();
            time &= 0x07_FF_FF_FFL;
            rank = type | time;
        }

        @Override
        public Address generateAddress() {
            return new CCSAddress(rank);
        }
        
    }
    
}
