package org.lsst.ccs.messaging.jgroups;

import java.util.*;
import org.jgroups.Address;
import org.jgroups.stack.MembershipChangePolicy;

/**
 * Custom JGroups membership policy for CCS.
 * Keeps suspects (currently disabled), sorts by address.
 *
 * @author onoprien
 */
public class DebugMembershipPolicy implements MembershipChangePolicy {
    
// -- Fields : -----------------------------------------------------------------

    static private final boolean log = "true".equalsIgnoreCase(System.getProperty("ccs.jg.MembershipPolicy.log"));
    static private final boolean keepSuspects = "true".equalsIgnoreCase(System.getProperty("ccs.jg.MembershipPolicy.keepSuspects"));
    
// -- Implement MembershipChangePolicy : ---------------------------------------
    
    @Override
    public List<Address> getNewMembership(Collection<Address> current_members,
                                          Collection<Address> joiners,
                                          Collection<Address> leavers,
                                          Collection<Address> suspects) {
        
        List<Address> out = keepSuspects ?
                (new Membership()).add(current_members).add(joiners).add(suspects).remove(leavers).sort().getMembers() :
                (new Membership()).add(current_members).add(joiners).remove(leavers).remove(suspects).sort().getMembers();
        if (log) {
            StringBuilder sb = new StringBuilder("TestMembershipPolicy:\n");
            sb.append("Current: ");
            current_members.forEach(m -> sb.append(m).append(" "));
            sb.append("\njoiners: ");
            joiners.forEach(m -> sb.append(m).append(" "));
            sb.append("\nleavers: ");
            leavers.forEach(m -> sb.append(m).append(" "));
            sb.append("\nsuspects: ");
            suspects.forEach(m -> sb.append(m).append(" "));
            sb.append("\nResult: ");
            out.forEach(m -> sb.append(m).append(" "));
            JGroupsBusMessagingLayer.getLogger().fine(sb.toString());
        }
        return out;
    }

    @Override
    public List<Address> getNewMembership(Collection<Collection<Address>> subviews) {
        Membership m = new Membership();
        subviews.forEach(view -> m.add(view));
        List<Address> out = m.sort().getMembers();
        if (log) {
            StringBuilder sb = new StringBuilder("TestMembershipPolicy merge:\n");
            subviews.forEach(v ->{
                sb.append("[ ");
                v.forEach(mm -> sb.append(mm).append(" "));
                sb.append(" ]\n");
            });
            sb.append("\nResult: ").append("[ ");
            out.forEach(mm -> sb.append(mm).append(" "));
            sb.append(" ]");
            JGroupsBusMessagingLayer.getLogger().fine(sb.toString());
        }
        return out;
    }
    
// -- Local classes : ----------------------------------------------------------
    
    /**
     * Replacement for org.jgroups.Membership.
     */
    static private final class Membership {
        
        private final ArrayList<Address> data = new ArrayList<>(128);
        
        Membership add(Address a) {
            if (a != null && !data.contains(a)) {
                data.add(a);
            }
            return this;
        }
        
        Membership add(Membership m) {
            if (m != null) {
                m.data.forEach(a -> add(a));
            }
            return this;
        }
        
        Membership add(Collection<Address> addresses) {
            if (addresses != null) {
                addresses.forEach(a -> add(a));
            }
            return this;
        }
        
        Membership remove(Collection<Address> aa) {
            if (aa != null) {
                data.removeAll(aa);
            }
            return this;
        }
        
        Membership sort() {
            data.sort((a1, a2) -> {
                if (a1 instanceof CCSAddress) {
                    if (a2 instanceof CCSAddress) {
                        return Long.compare(((CCSAddress) a1).getRank(), ((CCSAddress) a2).getRank());
                    } else {
                        return -1;
                    }
                } else {
                    if (a2 instanceof CCSAddress) {
                        return 1;
                    } else {
                        return a1.compareTo(a2);
                    }
                }
            });
            return this;
        }
        
        List<Address> getMembers() {
            return new ArrayList<>(data);
        }
        
    }
    
}
