package org.lsst.ccs.messaging.jgroups;

import java.util.*;
import org.jgroups.Address;
import org.jgroups.stack.MembershipChangePolicy;

/**
 * Custom JGroups membership policy for CCS.
 * Keeps suspects (currently disabled), sorts by address.
 *
 * @author onoprien
 */
public class CCSMembershipPolicy implements MembershipChangePolicy {

    @Override
    public List<Address> getNewMembership(Collection<Address> current_members,
                                          Collection<Address> joiners,
                                          Collection<Address> leavers,
                                          Collection<Address> suspects) {
        return (new Membership()).add(current_members).add(joiners).remove(leavers).remove(suspects).sort().getMembers(); // JGroups default (aside from sorting)
//        return (new Membership()).add(current_members).add(joiners).add(suspects).remove(leavers).sort().getMembers(); // keep suspects
    }

    @Override
    public List<Address> getNewMembership(Collection<Collection<Address>> subviews) {
        Membership m = new Membership();
        subviews.forEach(view -> m.add(view));
        return m.sort().getMembers();
    }
    
// -- Local classes : ----------------------------------------------------------
    
    /**
     * Replacement for org.jgroups.Membership.
     */
    static private final class Membership {
        
        private final ArrayList<Address> data = new ArrayList<>(128);
        
        Membership add(Address a) {
            if (a != null && !data.contains(a)) {
                data.add(a);
            }
            return this;
        }
        
        Membership add(Membership m) {
            if (m != null) {
                m.data.forEach(a -> add(a));
            }
            return this;
        }
        
        Membership add(Collection<Address> addresses) {
            if (addresses != null) {
                addresses.forEach(a -> add(a));
            }
            return this;
        }
        
        Membership remove(Collection<Address> aa) {
            if (aa != null) {
                data.removeAll(aa);
            }
            return this;
        }
        
        Membership sort() {
            data.sort((a1, a2) -> {
                if (a1 instanceof CCSAddress) {
                    if (a2 instanceof CCSAddress) {
                        return Long.compare(((CCSAddress) a1).getRank(), ((CCSAddress) a2).getRank());
                    } else {
                        return -1;
                    }
                } else {
                    if (a2 instanceof CCSAddress) {
                        return 1;
                    } else {
                        return a1.compareTo(a2);
                    }
                }
            });
            return this;
        }
        
        List<Address> getMembers() {
            return new ArrayList<>(data);
        }
        
    }
    
}
