package org.lsst.ccs.localdb.statusdb;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister<T> implements Runnable, HasLifecycle {

    protected final Logger log = Logger.getLogger(getClass().getCanonicalName());

    @LookupName
    private String persisterName;

    protected final Queue<T> rq = new ConcurrentLinkedQueue<>();
    
    @LookupField(strategy = Strategy.TREE)
    protected AlertService alertService;
    
    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;

    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;
    
    private volatile boolean drainQueue = false;
    
    //These two fields control how many threads we are using and the size
    //of the queue. For the FastBatchPersister, these are build configuration 
    //parameters, while they are not for the SlowBatchPersister.
    protected int ingThreadsPoolSize = 1;
    protected int ingQueueSize = 1;
        
    @ConfigurationParameter(category = "General", description = "Maximum number of entities to ingest in a single database transaction.")
    private volatile int maxIngestionSize = 1000;
    
    private ExecutorService exec;

    private boolean slowProcessing = false;

    BatchPersister() {
        this(false);
    }
    
    BatchPersister(boolean slowProcessing) {
        this.slowProcessing = slowProcessing;
    }
    
    private volatile AlertState batchSubmissionAlertState = AlertState.NOMINAL;
    
    public static ExecutorService newFixedThreadPoolBoundedQueue(int nThreads, int nQueue) {
        LinkedBlockingQueue queue = nQueue > 0 ? new LinkedBlockingQueue<>(nQueue) : new LinkedBlockingQueue<>();
        return new ThreadPoolExecutor(nThreads, nThreads, 0, TimeUnit.MILLISECONDS, queue);
    }
    
    @Override
    public void shutdown() {
        drainQueue = true;
        rq.clear();
        exec.shutdownNow();
        
        Session sess = StatusdbUtils.getSessionFactory().openSession();
        sess.setFlushMode(FlushMode.COMMIT);
        Transaction tx = null;
        try {
            tx = sess.beginTransaction();
            flush(sess);
            tx.commit();
        } catch (Exception ex) {
            log.log(Level.SEVERE, persisterName+" caught exception when performing shutdown flush", ex);
        } finally {
            if (sess.isOpen()) {
                sess.close();
            }
        }
    }

    @Override
    public void build() {
        log.log(Level.INFO,"Building Executor for {0} with {1} threads and pool size of {2}", new Object[]{persisterName, ingThreadsPoolSize, ingQueueSize} );
        exec = newFixedThreadPoolBoundedQueue(ingThreadsPoolSize, ingQueueSize);
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask(name, this).withIsFixedRate(false).withPeriod(Duration.ofMillis(100)));
    }    
    
    
    @Override
    public void init() {
        
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertCode.CLEAR_ALERT;
            }

        };
        
        alertService.registerAlert(LocalDBAlert.BatchException.getAlert(persisterName, null), alwaysClear);
        alertService.registerAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, null), alwaysClear);
        alertService.registerAlert(LocalDBAlert.BatchIngestionQueueSize.getAlert(persisterName, null), alwaysClear);
    }


    private void persistBatch(List<T> batch) {
        log.log(Level.FINER, "{0} start batch {1} {2} {3}", new Object[]{persisterName, this.getClass().getName(), this, batch.size()});
        long tstart = System.currentTimeMillis();
        Session sess = StatusdbUtils.getSessionFactory().openSession();

        sess.setFlushMode(FlushMode.COMMIT);

        Transaction tx = null;
        try {
            tx = sess.beginTransaction();
            for (T t : batch) {
                persist(t, sess);
            }
            tx.commit();
        } catch (Exception ex) {
            log.log(Level.SEVERE, persisterName+" caught exception when persisting", ex);
            alertService.raiseAlert(LocalDBAlert.BatchException.getAlert(persisterName, ex), AlertState.WARNING,
                    LocalDBAlert.getFirstException(ex));
            try {
                if (tx != null && (tx.getStatus() == TransactionStatus.ACTIVE
                        || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                    tx.rollback();
                }
            } catch (Exception rbEx) {
                log.log(Level.SEVERE, persisterName+" Rollback of transaction failed : " + rbEx, rbEx);
                alertService.raiseAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, ex),
                        AlertState.WARNING, LocalDBAlert.getFirstException(ex));
            }
        } finally {
            if (sess.isOpen()) {
                sess.close();
            }
        }

        long time = System.currentTimeMillis() - tstart;
        if (influxDbClientService.isEnabled() ) {
            Builder batchPersisterPointBuilder = Point.measurement("db_persist").time(System.currentTimeMillis(),
                    TimeUnit.MILLISECONDS);

            batchPersisterPointBuilder = batchPersisterPointBuilder.addField("btime_tot", time)
                    .addField("bcount", batch.size()).addField("btime_avg", (double) time / (double) batch.size());
            Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
            influxDbClientService.write(point);
        } 

        log.log(Level.FINER, "{0} end batch ", persisterName);
    }

    @Override
    public void run() {
        log.log(Level.FINE, "{0} start run {1}", new Object[]{persisterName, rq.size()});
        int n = 0;
        int nAcc = 0;
        boolean done = false;
        long tstart = System.currentTimeMillis();
        while (!done) {
            List<T> batch = new ArrayList(maxIngestionSize);
            // get at most nMax objets from the queue
            for (int i = 0; i < maxIngestionSize; i++) {
                T t = rq.poll();
                n = i;
                if (t == null) {
                    done = (i == 0);
                    break;
                } else {
                    batch.add(t);
                }
            }
            if (batch.size() > 0) {
                try {
                    
                    Future f = exec.submit(() -> persistBatch(batch));
                    if ( slowProcessing ) {
                        f.get();
                    }
                    
                    //Check the size of the queue containing the batch tasks only
                    //if this is not a slowProcessing instance.
                    //If it exceeds 70% of the maximum capacity 
                    //then raise an Alert at WARNING if we are currently in NOMINAL.
                    //Otherwise check if we are recovering and lower the state 
                    //back to NOMINAL if the queue size is below 50% of max capacity
                    if ( ! slowProcessing ) {
                        int taskSubmissionQueueSize = ((ThreadPoolExecutor) exec).getQueue().size();

                        String cause = null;
                        if (taskSubmissionQueueSize > ingQueueSize * .7 && batchSubmissionAlertState == AlertState.NOMINAL) {
                            //We are starting to slow down the ingestion
                            cause = "The submission queue is filling up. Ingestion is falling behind.\n";
                            batchSubmissionAlertState = AlertState.WARNING;
                        } else if (taskSubmissionQueueSize < ingQueueSize * .5 && batchSubmissionAlertState != AlertState.NOMINAL) {
                            cause = "The submission queue is being drained. Ingestion is catching up.\n";
                            batchSubmissionAlertState = AlertState.NOMINAL;
                        }
                        if (cause != null) {
                            Alert batchSubmissionAlert = LocalDBAlert.BatchIngestionQueueSize.getAlert(persisterName, null);
                            cause += "Executor task queue size: " + taskSubmissionQueueSize + " (nThreads=" + ingThreadsPoolSize + ")";
                            alertService.raiseAlert(batchSubmissionAlert, batchSubmissionAlertState, cause);
                        }
                    }
                } catch (RejectedExecutionException | InterruptedException | ExecutionException e) {
                    if ( batchSubmissionAlertState != AlertState.ALARM ) {                    
                        String cause = "Failed to submit a new task. Ingestion is all filled up. \nExecutor task queue size "+
                            + ((ThreadPoolExecutor) exec).getQueue().size();
                        batchSubmissionAlertState = AlertState.ALARM;
                        Alert batchSubmissionAlert = LocalDBAlert.BatchIngestionQueueSize.getAlert(persisterName, null);
                        alertService.raiseAlert(batchSubmissionAlert,batchSubmissionAlertState, cause);
                    }
                    //Put the batch into the queue for reprocessing 
                    rq.addAll(batch);
                }
                done = (rq.size() < maxIngestionSize / 5) || slowProcessing; // too few objects left for a
                                               // batch, we will wait for the
                                               // next round
                if (done)
                    log.log(Level.FINE, "{0} run done, left in queue {1}", new Object[]{persisterName, rq.size()});
            }

            if (n > 0) {
                nAcc += n;
                log.log(Level.FINER, "{0} processed {1} entities.", new Object[]{persisterName, n});
            }
        }
        if (nAcc > 0)

        {
            long time = System.currentTimeMillis() - tstart;
            if (influxDbClientService.isEnabled()) {
                Builder batchPersisterPointBuilder = Point.measurement("db_persist")
                        .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                int submissionQueueSize = ((ThreadPoolExecutor) exec).getQueue().size();
                double perThreadSize = (double)submissionQueueSize/ingThreadsPoolSize;

                batchPersisterPointBuilder = batchPersisterPointBuilder.
                        addField("time_tot", time).
                        addField("count", nAcc).
                        addField("time_avg", (double)time/(double)nAcc)
                        .addField("exec_queue_size", (double) submissionQueueSize).addField("exec_queue_size_per_threads", perThreadSize)
                        .addField("proc_queue_size", rq.size());
                Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
                influxDbClientService.write(point);                
            } 
        }
        log.log(Level.FINE, "{0} end run ", persisterName);
    }

    public void addToQueue(T obj) {
        if ( !drainQueue ) {
            rq.add(obj);
        }
    }

    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(T obj, Session sess);

    /**
     * Get the batch processing period.
     * 
     */
    public Duration getBatchProcessingPeriod() {
        return periodicTaskService.getPeriodicTaskPeriod(name);
    }
    
    /**
     * Invoked during shutdown to flush leftover data.
     * @param sess 
     */
    public abstract void flush(Session sess);
    
    
}
