package org.lsst.ccs.localdb.statusdb;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import org.hibernate.Query;
import org.hibernate.Session;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertInstance;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.localdb.dao.LocaldbFacade;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.AlertData;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.ClearedAlertData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.localdb.statusdb.model.StateChangeNotificationData;
import org.lsst.ccs.localdb.statusdb.model.StatusMessageData;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author LSST CCS Team
 */
public class StatusPersister extends FastBatchPersister<Object[]> implements StatusMessageListener, AgentPresenceListener {
    
    /**
     * COMMENTS REGARDING CLEANING UP DATABASE TABLES
     * 
     * When the local database starts up there might be some database cleaning up 
     * that needs to be performed in the RaisedAlertData table.
     * 
     * When an agent is *connecting* we add its name to the list below of agentsToCheck.
     * The first time we receive a StatusRaisedAlertSummary (published spontaneously by
     * the connecting agent) we get the timestamp of the earliest active Alert and
     * mark as disabled any RaisedAlert before such timestamp.
     */
    private final List<String> agentsToCheck = new CopyOnWriteArrayList<>();
    
    @LookupField(strategy=Strategy.TOP)
    private Agent subsys;
    
    @Override
    public void init() {
        super.init();
        
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }

        };
        
        alertService.registerAlert(LocalDBAlert.BatchException.getAlert(null, null), alwaysClear);
    }
    
    @Override
    public void start() {
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        subsys.getMessagingAccess().addStatusMessageListener(this,
                BusMessageFilterFactory.messageOrigin(null).and(
                        BusMessageFilterFactory.messageClass(StatusAlert.class)
                                .or(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class)
                                        .and( bm -> bm.getOriginAgentInfo().getType().compareTo(AgentInfo.AgentType.WORKER) >=0
                                        ))
                ));
    }
    
    @Override
    public void onStatusMessage(StatusMessage msg) {
        if (msg instanceof StatusAlert) {
            
            StatusAlert statusAlert = (StatusAlert)msg;
            // Case 1 : received a StatusRaisedAlert
            if (statusAlert instanceof StatusRaisedAlert) {
                StatusRaisedAlert statusRaisedAlert = (StatusRaisedAlert) statusAlert;
                String alertId = ((Alert)statusAlert.getEncodedData()).getAlertId();
                String alertDescription = statusRaisedAlert.getCause();
                AlertState currentSeverity = statusRaisedAlert.getRaisedAlertState();
                RaisedAlertData arad = new RaisedAlertData(currentSeverity, alertDescription, msg.getCCSTimeStamp().getUTCInstant().toEpochMilli());
                addToQueue(new Object[]{arad, statusAlert, alertId});
            } else if (statusAlert instanceof StatusClearedAlert) {
                // Case 2 : received a StatusClearedAlert
                StatusClearedAlert sca = (StatusClearedAlert)statusAlert;
                String[] clearedAlerts = sca.getClearAlertIds();
                if (clearedAlerts != null) {
                    for (String clearedId : clearedAlerts) {
                        ClearedAlertData cad = new ClearedAlertData();
                        cad.setTime(msg.getCCSTimeStamp().getUTCInstant().toEpochMilli());
                        addToQueue(new Object[]{cad, statusAlert,clearedId});
                    }
                }
            } else if (statusAlert instanceof StatusRaisedAlertSummary ) {
                //Case 3: it's a StatusRaisedAlertSummary
                
                //If this is the first StatusRaisedAlertSummary
                //submit it to the queue for processing
                if ( agentsToCheck.contains(statusAlert.getOriginAgentInfo().getName()) ) {
                    addToQueue(new Object[]{msg, null});
                }
            }
        } else if (msg instanceof StatusStateChangeNotification) {
            StatusStateChangeNotification statusSCN = (StatusStateChangeNotification)msg;
            StateChangeNotificationData scnd = new StateChangeNotificationData();
            scnd.setTime(msg.getCCSTimeStamp().getUTCInstant().toEpochMilli());
            addToQueue(new Object[]{scnd,statusSCN});
        } 
    }
    
    @Override
    public void connecting(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.getType().compareTo(AgentType.WORKER) >= 0) {
                addToQueue(new Object[]{new AgentPresenceEvent(agent, System.currentTimeMillis(), true)});
                agentsToCheck.add(agent.getName());
            }
        }
    }
    
    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            if ( agent.getType().compareTo(AgentType.WORKER) >= 0 ) {           
                addToQueue(new Object[]{new AgentPresenceEvent(agent, System.currentTimeMillis(), false)});
            }
        }
    }
    
    private void pushActiveAlerts(String agentName, long agentStartTime, Session sess ) {
        agentsToCheck.remove(agentName);
        Query q = sess.createQuery("update RaisedAlertData rad set rad.active=FALSE where rad.agentDesc.agentName=:name and rad.active=true and rad.time<=:t");
        q.setString("name", agentName);
        q.setLong("t", agentStartTime);
        int nPushed = q.executeUpdate();
        if (nPushed > 0) {
            log.info("updated " + nPushed +" active alerts for agent : " + agentName);
        }
    }
    
    /**
     * Called from inside a transaction.
     * This method is called sequentially : LocaldbFacade needs not be thread safe.
     * TO-DO : is this the right strategy ?
     * @param obj
     * @param sess
     */
    @Override
    public void persist(Object[] obj, Session sess) {
        if(obj[0] instanceof StatusMessageData) {
            // Operations common to every status message
            StatusMessageData msgData = (StatusMessageData)obj[0];
            StatusMessage statusMsg = (StatusMessage)obj[1];
            
            AgentInfo ai = statusMsg.getOriginAgentInfo();
            StateBundle state = statusMsg.getState();
            
            // Fetching agent description from local cache or create it and persist it.
            AgentDesc agDesc = LocaldbFacade.getAgentDesc(ai, sess);
            msgData.setAgentDesc(agDesc);
            
            // State reconstruction.
            AgentState stateData = LocaldbFacade.getAgentState(agDesc, state, sess);
            msgData.setAgentState(stateData);
            
            // Operations specific to sub classes of status messages.
            if (msgData instanceof AlertData) {
                // The object to be persisted
                AlertData alData = (AlertData)msgData;
                
                // The original status
                StatusAlert statusAlert = (StatusAlert)statusMsg;
                
                // The alertId associated to
                String alertId = (String)obj[2];
                
                // Fetching alert description from local cache or create and persist it.
                if (statusAlert instanceof StatusRaisedAlert) {
                    Alert al = ((StatusRaisedAlert)statusAlert).getRaisedAlert();
                    AlertDesc ad = LocaldbFacade.getAlertDescOrPersist(agDesc, al, sess);
                    alData.setAlertDesc(ad);
                    sess.persist(alData);
                    
                } else if (statusAlert instanceof StatusClearedAlert) {
                    AlertDesc ad = LocaldbFacade.getAlertDesc(agDesc, alertId, sess);
                    if (ad != null) {
                        alData.setAlertDesc(ad);
                        sess.persist(alData);
                    } else {
                        log.log(Level.WARNING, "could not find description for alert with id {0}", alertId);
                    }
                    // Cleared flag is updated on the instances of the active cleared alert.
                    Query q = sess.createQuery("from RaisedAlertData rad where"
                            + " rad.agentDesc.agentName=:name and rad.alertDesc.alertId =:id and rad.active=true")
                            .setString("id", alertId).setString("name", agDesc.getAgentName());
                    List<RaisedAlertData> res = q.list();
                    for (RaisedAlertData arad : res) {
                        arad.setClearingAlert((ClearedAlertData)alData);
                        sess.update(arad);
                    }
                    log.log(Level.FINE, "cleared {0} instances of {1}/{2}", new Object[]{res.size(), agDesc.getAgentName(), alertId});
                }
            } else if (msgData instanceof StateChangeNotificationData) {
                StateChangeNotificationData scnd = (StateChangeNotificationData)msgData;
                
                StateBundle oldState = ((StatusStateChangeNotification)statusMsg).getOldState();
                if ( oldState == null ) {
                    log.log(Level.WARNING, "Null old state in StateChangeNotificationData for {0}", ai.getName());
                    return;
                }
                scnd.setOldState(LocaldbFacade.getAgentState(agDesc, oldState, sess));
                
                if(scnd.getOldState().getId() == scnd.getNewState().getId() && !state.equals(oldState)) {
                    log.log(Level.WARNING, "configuration persister considers old state and new state equal ({0}). Diff actually is :\n{1}", new Object[]{scnd.getOldState().getId(), state.diffState(oldState)});
                    alertService.raiseAlert(LocalDBAlert.WrongStatePersistence.addData("oldState", oldState).addData("newState", state).getAlert(null, null),AlertState.WARNING, "for agent " + ai.getName());
                }
                sess.persist(scnd);
            } 
        } else if (obj[0] instanceof AgentPresenceEvent) {
            AgentPresenceEvent ape = (AgentPresenceEvent)obj[0];
            if(!ape.isConnecting()) {
                pushActiveAlerts(ape.getAgent().getName(), ape.getTime(), sess);
            }
        } else if (obj[0] instanceof StatusRaisedAlertSummary ) {
            StatusRaisedAlertSummary summary = (StatusRaisedAlertSummary) obj[0];
            Set<RaisedAlertHistory> alertHistorySet = summary.getRaisedAlertSummary().getAllRaisedAlertHistories();
            CCSTimeStamp earliestAlertTimestamp = CCSTimeStamp.currentTime();
            for (RaisedAlertHistory alertHistory : alertHistorySet) {
                //The earlier alert is the first element of the raised alert list
                RaisedAlertInstance earliestRaiseAlert = alertHistory.getRaisedAlertInstancesList().get(0);
                if (earliestAlertTimestamp.getUTCInstant().isAfter(earliestRaiseAlert.getCCSTimeStamp().getUTCInstant())) {
                    earliestAlertTimestamp = earliestRaiseAlert.getCCSTimeStamp();
                }
            }
            //To avoid clearing the earliest alert, we subctract 1ms from its UTC instant
            Instant instant = earliestAlertTimestamp.getUTCInstant().minusMillis(1);
            pushActiveAlerts(summary.getOriginAgentInfo().getName(), instant.getEpochSecond(), sess);
        }

    }
}
