package org.lsst.ccs.localdb.statusdb;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.hibernate.Session;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.localdb.dao.LocaldbFacade;
import static org.lsst.ccs.localdb.statusdb.StatusDataPersister.getStatDescs;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;

/**
 * Batch persister for {@code StatData} related entities.
 * @author LSST CCS Team
 */
public class StatDataAccumulator extends FastBatchPersister<RawData> {
        
    @LookupField(strategy=LookupField.Strategy.TREE)
    private StatDataPersister statDataPersister;
    
    // a cache of statData from transient to persistent state
    private final Map<DataDesc, Map<StatDesc, StatData>> statCache = new HashMap<>();
        
    @Override
    public void persist(RawData data, Session sess) {
        
        DataDesc dd = data.getDataDesc();
        Map<StatDesc, StatData> dataCache = statCache.get(dd);
        if (dataCache == null) {
            dataCache = new HashMap<>();
            statCache.put(dd, dataCache);
        }

        List<StatDesc> stats = getStatDescs(dd, sess, log);

        long dataTime = data.getTime();

        for (StatDesc stat : stats) {
            if (data.getDoubleData() == null) {
                continue;
            }

            // Statistical bin width
            long binWidth = stat.getTimeBinWidth();
            StatTimeInterval sti = LocaldbFacade.getStatTimeInterval(binWidth, dataTime);

            StatData cachedStat = dataCache.get(stat);
            if (cachedStat == null) {
                // No cached stat for this statDesc
                // Looking for a statdata in db
                StatData persistedSD = LocaldbFacade.getStatDataOrNull(stat, sti, sess);
                if (persistedSD != null) {
                    log.fine("found existing stat data for " + persistedSD.getStatDesc().getDataDesc().getDataPath().getFullKey());
                    persistedSD.accumulate(data);
                    dataCache.put(stat, persistedSD);
                } else {
                    // No statData in db neither : creating a brand new one
                    dataCache.put(stat, new StatData(stat, data.getDoubleData(), sti));
                }
            } else if (cachedStat.getStatTimeInterval().equals(sti)) {
                // The data falls into the cached statData
                cachedStat.accumulate(data);
            } else if (cachedStat.getStatTimeInterval().getStartTime() < sti.getStartTime()) {
                
                // the data falls into the next statData
                // The cached StatData becomes persistent

                // Submit here to be persisted
                statDataPersister.addToQueue(cachedStat);

                // The next StatData may already exist
                StatData newStatData = LocaldbFacade.getStatDataOrNull(stat, sti, sess);
                if (newStatData != null) {
                    newStatData.accumulate(data);
                } else {
                    newStatData = new StatData(stat, data.getDoubleData(), sti);
                }
                dataCache.put(stat, newStatData);
            } else {
                // Anomaly : the cached statData is ahead of the received data
                // TO-DO : do we need to flush
                log.warning("IGNORING data received in a non ascending timestamp order for key : " + dd.getDataPath().getFullKey());

//                // persist the cached data
//                // Submit here to be persisted
//                statDataPersister.addToQueue(cachedStat);
//
//                // Look in db for an existing statData
//                StatData statData = LocaldbFacade.getStatDataOrNull(stat, sti, sess);
//                if (statData != null) {
//                    log.fine("found existing stat data for " + statData.getStatDesc().getDataDesc().getDataPath().getFullKey());
//                    statData.accumulate(data);
//                    dataCache.put(stat, statData);
//                } else {
//                    // No statData in db neither : creating a brand new one
//                    // Do not persist it yet
//                    statData = new StatData(stat, data.getDoubleData(), sti);
//                }
//                dataCache.put(stat, statData);
            }
        }
    }
      
    /**
     * Persists all the cached statistical data.
     * @param sess
     */
    public void flush(Session sess) {
        for(Map.Entry<DataDesc, Map<StatDesc, StatData>> entry : statCache.entrySet()) {
            Iterator<Map.Entry<StatDesc, StatData>> it = entry.getValue().entrySet().iterator();
            // make all statData that use the now deprecated sti persistent
            while(it.hasNext()) {
                Map.Entry<StatDesc, StatData> e = it.next();
                StatData val = e.getValue();
                // Submit here to be persisted
                statDataPersister.addToQueue(val);
                it.remove();
            }
        }
        statDataPersister.flush(sess);
    }
    
    
    
    
}
