package org.lsst.ccs.localdb.utils;

import org.influxdb.BatchOptions;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.Pong;
import org.lsst.ccs.ServiceLifecycle;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.services.AgentService;

/**
 * An AgentService to be used by Agents that need to interact with the Rest
 * Server on the buses.
 * Connections can be established by either creating a custom connection to
 * a known host:port combination or by listening to the buses for available rest services.
 * 
 * This service is activated for any agent that specifies Bootstrap property
 * org.lsst.ccs.use.influxdb.service=true
 * 
 * @author The LSST CCS Team
 * 
 */
public class InfluxDbClientService implements AgentService, ServiceLifecycle {

    private static final java.util.logging.Logger LOG = java.util.logging.Logger.getLogger(InfluxDbClientService.class.getName());
    
    @ConfigurationParameter(isFinal = true, category = "InfluxDb")
    private String influxDbUrl;
    @ConfigurationParameter(isFinal = true, category = "InfluxDb")
    private String influxDbName;
    @ConfigurationParameter(isFinal = true, category = "InfluxDb")
    private String influxDbUsername;
    @ConfigurationParameter(isFinal = true, category = "InfluxDb")
    private String influxDbPassword;

    private InfluxDB influxDb;
    
    @Override
    public String getAgentServiceName() {
        return "influxDbClientService";
    }

    @Override
    public boolean startForAgent(AgentInfo agentInfo) {
        if ( BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.use.influxdb.service", "false").toLowerCase().equals("true") ) {
            LOG.info("Starting InfluxDbClientService for agent "+agentInfo.getName());
            return true;
        } 
        return false;
    }
    
    
    
    @Override
    public void preStart() {
        if (influxDbUrl != null && !influxDbUrl.isEmpty()) {
            influxDb = InfluxDBFactory.connect(influxDbUrl, influxDbUsername, influxDbPassword);
            influxDb = influxDb.setDatabase(influxDbName);
            Pong p = influxDb.ping();
            if (p == null || !p.isGood()) {
                throw new RuntimeException("Could not connect with InfluxDb " + p);
            }
            influxDb = influxDb.enableBatch(BatchOptions.DEFAULTS.actions(2000).flushDuration(500).exceptionHandler(
                    (failedPoints, throwable) -> {
                        LOG.severe(throwable.toString());
                    })
            );
        }
    }

    @Override
    public void preShutdown() {
        if (influxDb != null) {
            influxDb.close();
        }
    }
    
    public InfluxDB getInfluxDbClient() {
        return influxDb;
    }
}
