package org.lsst.ccs.efd;

import java.util.ArrayList;
import org.lsst.ccs.localdb.statusdb.server.*;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import javax.inject.Singleton;
import javax.ws.rs.Path;
import org.hibernate.Session;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.Pong;
import org.influxdb.dto.QueryResult;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;

/**
 * The DataServer receives requests for trending data analysis. It delivers the
 * requested data as the result of requests to the database.
 */
@SuppressWarnings("restriction")
@Path("/dataserver")
@Singleton
public class EfdDataServer extends DataServer {

    private InfluxDB influxDb;
    private String namespace;
    private static final Logger LOG = Logger.getLogger(EfdDataServer.class.getName());
    
    
    public EfdDataServer() {

        LOG.info("Created EFD Rest Server");
        getInfluxDb();
    }

    //Return the empty list so that no statistical data query is triggered
    @Override
    public List<StatDesc> getAvailableStats(long rawId, Session sess) {
        return Collections.EMPTY_LIST;
    }

    
    private InfluxDB getInfluxDb() {
        if ( influxDb == null ) {
            String credentialsFile = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.efd.database.connection.file", "efdCredentials.properties");
            Properties credentialsProperties = BootstrapResourceUtils.getBootstrapProperties(credentialsFile);
            String username = credentialsProperties.getProperty("org.lsst.ccs.efd.username", "");
            String password = credentialsProperties.getProperty("org.lsst.ccs.efd.password", "");
            String url = credentialsProperties.getProperty("org.lsst.ccs.efd.url", "");
            
            namespace = credentialsProperties.getProperty("org.lsst.ccs.efd.namespace", "");
            if (username.isEmpty() || password.isEmpty() || url.isEmpty() || namespace.isEmpty()) {
                throw new RuntimeException("The following properties must be specified in file " + credentialsFile + ": \norg.lsst.ccs.efd.username\norg.lsst.ccs.efd.password\norg.lsst.ccs.efd.url\norg.lsst.ccs.efd.napespace");
            }
            influxDb = InfluxDBFactory.connect(url, username, password);
        }
        return influxDb;
    }
    
    
    @Override
    public List<RawData> getRawData(String path, long t1, long t2, Session sess) {

        EfdQueryBuilder queryBuilder = new EfdQueryBuilder(namespace);
        queryBuilder = queryBuilder.forPath(path).withTimeRange(t1, t2);
        org.influxdb.dto.Query query = queryBuilder.buildQuery();
        
//        LOG.log(Level.WARNING,"EFD RawData Query: "+ query.getCommand());

        QueryResult queryResult = influxDb.query(query, TimeUnit.MILLISECONDS);

        if ( queryResult.getResults().size() > 1 ) {
            throw new RuntimeException("Got more results than expected: "+queryResult.getResults().size()+" for query "+query.getCommand());
        }
        
        List<RawData> dataList = null;
        for (QueryResult.Result r : queryResult.getResults()) {
            if ( r.getSeries().size() > 1 ) {
                throw new RuntimeException("Got more series than expected: "+r.getSeries().size()+" for query "+query.getCommand());
            }
            for (QueryResult.Series s : r.getSeries()) {
                for (String column : s.getColumns()) {
                    if (column.equals("time")) {
                        continue;
                    }
                    dataList = convertSeriesToRawData(s, column);
                }
            }
        }        
        return dataList;
    }

    protected List<StatEntry> getStatDataFromRaw(String path, long t1, long t2, int nsamples, Session sess) {

        EfdQueryBuilder queryBuilder = new EfdQueryBuilder(namespace);
        queryBuilder = queryBuilder.forPath(path).withTimeRange(t1, t2).withSamples(nsamples);
        org.influxdb.dto.Query query = queryBuilder.buildQuery();
//        LOG.log(Level.WARNING,"EFD StatData Query: " +query.getCommand());
        
        QueryResult queryResult = influxDb.query(query, TimeUnit.MILLISECONDS);

        if ( queryResult.getResults().size() > 1 ) {
            throw new RuntimeException("Got more results than expected: "+queryResult.getResults().size()+" for query "+query.getCommand());
        }
        
        List<StatEntry> dataList = Collections.EMPTY_LIST;
        for (QueryResult.Result r : queryResult.getResults()) {
            if ( r.getSeries() == null ) {
                continue;
            }
            if ( r.getSeries().size() > 1 ) {
                throw new RuntimeException("Got more series than expected: "+r.getSeries().size()+" for query "+query.getCommand());
            }
            for (QueryResult.Series s : r.getSeries()) {
                for (String column : s.getColumns()) {
                    dataList = convertSeriesToStatData(s, queryBuilder.getTimeWidth());
                }
            }
        }        
        return dataList;

    }
    
    
    @Override
    public List<StatData> getStatData(StatDesc statDesc, long t1, long t2, Session sess) {
        throw new UnsupportedOperationException("Stat data queries are not supported for the EFD Database");
    }
    
    
    public static List<RawData> convertSeriesToRawData(QueryResult.Series series, String column) {
        List<RawData> r = new ArrayList<>();
        int timeIndex = series.getColumns().indexOf("time");
        int dataIndex = series.getColumns().indexOf(column);
        
        for ( List<Object> dataList : series.getValues() ) {
            Object time = dataList.get(timeIndex);
            Object data = dataList.get(dataIndex);
            if ( data == null )
                continue;
            RawData rawData = new RawData();
            rawData.setTime(((Double)time).longValue());
            rawData.setDoubleData((double)data);
            r.add(rawData);
        }        
        return r;        
    }

    public static List<StatEntry> convertSeriesToStatData(QueryResult.Series series, long timeWidth) {
        List<StatEntry> r = new ArrayList<>();
        int timeIndex = series.getColumns().indexOf("time");
        int maxValueIndex = series.getColumns().indexOf("maxValue");
        int minValueIndex = series.getColumns().indexOf("minValue");
        int meanValueIndex = series.getColumns().indexOf("meanValue");
        int stddevValueIndex = series.getColumns().indexOf("stddevValue");
        
        for ( List<Object> dataList : series.getValues() ) {
            Object time = dataList.get(timeIndex);
            Object maxValue = dataList.get(maxValueIndex);
            if ( maxValue == null )
                continue;
            Object minValue = dataList.get(minValueIndex);
            if ( minValue == null )
                continue;
            Object meanValue = dataList.get(meanValueIndex);
            if ( meanValue == null )
                continue;
            Object stddevValue = dataList.get(stddevValueIndex);
            double stddev = 0;
            if ( stddevValue != null ) 
                stddev = (Double)stddevValue;
            long minTime = ((Double)time).longValue();
            long maxTime = minTime + timeWidth;
            
            StatEntry statEntry = new StatEntry(minTime, maxTime, (Double)meanValue, stddev, (Double)minValue, (Double)maxValue);
            
            r.add(statEntry);
        }        
        return r;        
    }
    
    
    
    public static void main(String[] args) throws Exception {

        String additionalResourceDirectories =  "~/ccs/etc";
        System.setProperty("org.lsst.ccs.resource.path", additionalResourceDirectories);

        EfdDataServer dataServer = new EfdDataServer();
        InfluxDB influxDb = dataServer.getInfluxDb();

        Pong p = influxDb.ping();
        System.out.println(p);
        try {
            
            
            EfdQueryBuilder queryBuilder = new EfdQueryBuilder("lsst.MTCamera");
            queryBuilder = queryBuilder.forPath("chiller1/Chiller/TempStage2Evap").withTimeRange(1743005901693L, 1743006501693L)

//                    .withTimeRange(System.currentTimeMillis() - 10*60000L, System.currentTimeMillis())
                    .withSamples(100);
                    ;
            org.influxdb.dto.Query query = queryBuilder.buildQuery();
            System.out.println(query.getCommand()+" "+query.getDatabase());
            
//            Query query = queryBuilder.buildQuery();

//            Query query = new Query("select mean(publish_time_max) from kafka_proxy_metrics where time > now() - 1m and environment = 'camera' group by time(1m), topic", "grafana");
//            Query query = new Query("select publish_time_max from kafka_proxy_metrics where time > now() - 1m and environment = 'camera' and topic = 'focal_plane_reb_data'", "grafana");
//            Query query = new Query("show measurements","lsst.MTCamera");
//           Query query = new Query("select TempTXVBulb, TempCondenser from \"lsst.MTCamera.chiller\" where time > now() - 1m and Agent = 'chiller1'","lsst.MTCamera");

//            Query query = new Query("select count(1) from 'lsst.MTCamera.chiller'","lsst.MTCamera");

            QueryResult queryResult = influxDb.query(query, TimeUnit.MILLISECONDS);

            for (QueryResult.Result r : queryResult.getResults()) {
                for ( QueryResult.Series s : r.getSeries()) {                    
                    System.out.println("**** "+s.toString().replaceAll("], ", "]\n").replaceAll("values=","").replace("[", "").replace("]", ""));
                    if (queryBuilder.isRawData()) {
                        for (String column : s.getColumns()) {
                            if (column.equals("time")) {
                                continue;
                            }
                            List<RawData> data = convertSeriesToRawData(s, column);
                            System.out.println(data);
                        }
                    } else {
                        List<StatEntry> dataList = convertSeriesToStatData(s, queryBuilder.getTimeWidth());
                        System.out.println("Got "+dataList.size()+" statistical entries");
                        System.out.println(dataList);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            influxDb.close();
            System.exit(0);
        }

    }


}
