package org.lsst.ccs.localdb.statusdb.model;

import java.io.Serializable;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Index;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.QueryHint;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;

@Entity
@Table(
        uniqueConstraints = @UniqueConstraint(columnNames = {"statDescId","statTimeIntervalId"}),
        indexes = {@Index(columnList = "statDescId, statTimeIntervalId")}
)
@Cacheable
@org.hibernate.annotations.Cache( 
        usage = org.hibernate.annotations.CacheConcurrencyStrategy.READ_WRITE
        ,region = "org.lsst.ccs.localdb.statusdb.model.StatData"
)
@NamedQueries ( {
    @NamedQuery (
            name = "findStatData",
            query = "from StatData s where s.statDesc=:d and s.statTimeInterval=:sti",
            hints = {@QueryHint(name="org.hibernate.cacheable", value="true")}
    )
})
public class StatData implements Serializable {

    static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");
    private static final long serialVersionUID = -8845106173875848941L;
    private long id;
    private StatDesc statDesc;
    private StatTimeInterval timeInterval;
    private double average; // the average
    private double sum2;
    private int n;
    private double minval;
    private double maxval;

    public StatData() {
    }

    public StatData(StatDesc d, RawData x, StatTimeInterval timeInterval) {
        log.fine("new StatData from " + d.getDataDesc().getDataPath().getFullKey() + " : " + x.getDoubleData() + " at " + x.getTime());
        statDesc = d;
        average = x.getDoubleData();
        sum2 = average * average;
        minval = average;
        maxval = average;
        n = 1;
        this.timeInterval = timeInterval;
    }

    @Transient
    public double getSum() {
        return getAverage() * getN();
    }

    public void setSum(double s) {
        setAverage(s / getN());
    }

    @Transient
    public double getStdDev() {
        return Math.sqrt(getSum2() / getN() - getAverage() * getAverage());
    }

    public void accumulate(RawData x) {
        double sum = getSum();
        double d = x.getDoubleData();
        if ( !Double.isFinite(d) ) {
            log.log(Level.WARNING, "Ignoring non finite value {0} for channel {1}",new Object[]{d,x.getDataDesc().getDataPath().getFullKey()});
            return;
        }
        sum += d;
        setSum2(getSum2() + d * d);
        if (getN() == 0 || d > maxval)
            maxval = d;
        if (getN() == 0 || d < minval)
            minval = d;
        setN(getN() + 1);
        setSum(sum);
    }
    
    /**
     * @return the id
     */
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    public long getId() {
        return id;
    }

    /**
     * @param id
     *            the id to set
     */
    public void setId(long id) {
        this.id = id;
    }

    /**
     * @return the statDesc
     */
    @ManyToOne(optional = false, cascade = CascadeType.MERGE)
    @JoinColumn(name = "statDescId", updatable = false)
    public StatDesc getStatDesc() {
        return statDesc;
    }

    /**
     * @param statDesc
     *            the statDesc to set
     */
    public void setStatDesc(StatDesc statDesc) {
        this.statDesc = statDesc;
    }

    /**
     * Get the StatTimeInterval for this StatData object.
     * 
     * @return The StatTimeInterval.
     */
    @ManyToOne(optional = false, cascade = CascadeType.MERGE)
    @JoinColumn(name = "statTimeIntervalId",updatable = false)
    public StatTimeInterval getStatTimeInterval() {
        return timeInterval;
    }
    
    /**
     * Set the StatTimeInterval corresponding to this object.
     * @param timeInterval The StatTimeInterval.
     */
    public void setStatTimeInterval(StatTimeInterval timeInterval) {
        this.timeInterval = timeInterval;
    }
    
    /**
     * @return the average
     */
    public double getAverage() {
        return average;
    }

    /**
     * @param average
     *            the average to set
     */
    public void setAverage(double average) {
        this.average = average;
    }

    /**
     * @return the sum2
     */
    public double getSum2() {
        return sum2;
    }

    /**
     * @param sum2
     *            the sum2 to set
     */
    public void setSum2(double sum2) {
        this.sum2 = sum2;
    }

    /**
     * @return the n
     */
    public int getN() {
        return n;
    }

    /**
     * @param n
     *            the n to set
     */
    public void setN(int n) {
        this.n = n;
    }
    
    public void setMin(double min) {
        this.minval = min;
    }

    @Column(name = "minval")
    public double getMin() {
        return minval;
    }
    
    public void setMax(double max) {
        this.maxval = max;
    }

    @Column(name = "maxval")
    public double getMax() {
        return maxval;
    }
}
