package org.lsst.ccs.localdb.configdb.model;


import java.io.Serializable;
import java.util.Objects;
import org.hibernate.annotations.Immutable;

import javax.persistence.Embeddable;
import javax.persistence.Transient;

/**
 * Unique Path in a SubSystem. Objects of this class are immutable.
 * (no public mutator).
 */
@Embeddable
@Immutable
public class ParameterPath implements PathObject, Serializable {
    private static final long serialVersionUID = -1012785409109930223L;
    
    /**
     * unique name of Module/component in subsystem context
     */
    private String componentName;

    /**
     * name of parameter (may be the number of a positional parameter starting at 0)
     */
    private String parameterName ;
    
    //////////////////////////////////// CONSTRUCTORS

    /**
     * non public empty constructor
     */
    protected ParameterPath() {
    }

    /**
     * @param componentName should not be null
     * @param parameterName should not be null
     */
    public ParameterPath(String componentName, String parameterName) {
        if(componentName == null || parameterName == null) {
            throw new IllegalArgumentException("null argument in ParameterPath");
        }
        this.componentName = componentName;
        this.parameterName = parameterName;
    }
    
    ////////////////////////////////// ACCESSORS/MUTATORS    
    public String getComponentName() {
        return componentName;
    }

     void setComponentName(String componentName) {
        this.componentName = componentName;
    }

    public String getParameterName() {
        return parameterName;
    }

     void setParameterName(String parameterName) {
        this.parameterName = parameterName;
    }
    
    ////////////////////////////// IDENT METHODS

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ParameterPath that = (ParameterPath) o;

        if (!componentName.equals(that.componentName)) return false;
        if (!parameterName.equals(that.parameterName)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 79 * hash + Objects.hashCode(this.componentName);
        hash = 79 * hash + Objects.hashCode(this.parameterName);
        return hash;
    }
    
    @Override
    public String toString() {
        return this.componentName + '/' +this.parameterName ;
    }

    /**
     * reverse operation from toString(): creates a ParameterPath from a String.
     * @param pathString should be of the form "componentName/codeName/parameterName" (example: "carousel/tickMillis")
     * @return a ParameterPath out of the given path String
     * @throws  IllegalArgumentException if format is not correct
     */
    public static ParameterPath valueOf(String pathString) {
        ParameterPath path ;
        String[] elements = pathString.split("/") ;
        if(elements.length != 2) {
            throw new IllegalArgumentException("PathString should be componentName/parameterName");
        }
        path = new ParameterPath(elements[0], elements[1]) ;
        return path ;
    }

    @Transient
    @Override
    public ParameterPath getPath() {
        return this;
    }

}
