package org.lsst.ccs.localdb.configdb.model;

import org.hibernate.annotations.Immutable;

import javax.persistence.*;
import java.io.Serializable;
import java.util.*;

/**
 * An active subsystem Description.
 * Can be "unregistered" (id is zero) or "registered" in the database (Id is not zero)
 * in that case the object is immutable (more precisely only comments can be modified).
 *
 * Subsystem name and tag constitute a unique key in the active  table.
 *
 * @author bamade
 */
// Date: 10/04/12
@Table(
        uniqueConstraints = {@UniqueConstraint(columnNames={"subsystemName", "tag"})}
)
@Entity
@Immutable
public class ASubsystemDescription  extends SubsystemDescription implements Cloneable{
    private static final long serialVersionUID = 7518445583831244327L;
    @Id
    @GeneratedValue
    private long id ; //generated

    // TODO: annotate accessor method instead to avoid hack about unmodifiable parms
    /**
     * the set containing the parameter descriptions
     */
    @OneToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    protected Map<ParameterPath,AParameterDescription> paramDescriptions = new HashMap<>();
    
    // A set of all of the categories
    @Transient
    private Set<String> categorySet = new HashSet<String>();
    /////////////////////////////// CONSTRUCTORS

    protected ASubsystemDescription(){
    }

    /**
     * same parameters and constraints as the superclass constructor
     * @param subsystemName
     * @param tag
     * @param configurationData
     */
    public ASubsystemDescription(String subsystemName, String tag, Serializable configurationData) {
        super(subsystemName, tag, configurationData);
    }

    /**
     * creates a new SubsystemDescription without information set up by the database (id, startDate)
     * @param other
     */
    public ASubsystemDescription(SubsystemDescription other) {
        super(other) ;
        // now fill
        for(ParameterDescription parmDesc: other.getParamDescriptionMap().values()) {
            this.addParameterDescriptions(new AParameterDescription(parmDesc));
        }
    }


    ////////////////////////////// ACCESSORS/MUTATORS

    //    @Override
    public long getId() {
        return id;
    }

    @Override
    void setId(long id) {
        this.id = id ;
    }

    @Override
    public Map<ParameterPath,? extends ParameterDescription> getParamDescriptionMap() {
        return paramDescriptions;
    }
    
    @Override
    public Set<String> getCategorySet(){
        return Collections.unmodifiableSet(categorySet);
    }
    
    //@Generated(value = GenerationTime.INSERT)
    //@Temporal(TemporalType.DATE)
    @Override
    public long getStartTimestamp() {
        return super.getStartTimestamp();
    }


    /**
     * a data in this set is modifiable only if the object is not registered.
     * Use methods <TT>addParameterDescriptions</TT> and <TT>removeParameterDescriptions</TT>
     * to modify the set.
     * @return an unmodifiable set
     */
     Collection<AParameterDescription> getParamDescriptions() {
        return this.paramDescriptions.values() ;
    }
    
    void setParamDescriptions(Map<ParameterPath, AParameterDescription> paramDescriptions) {
        this.paramDescriptions = paramDescriptions;
    }

    /**
     * clone: the contained set of parameter description is also cloned.
     * (each description is cloned)
     * @return a clone of the object
     */
    public ASubsystemDescription clone() {
        ASubsystemDescription res = null ;
        try {
            res = (ASubsystemDescription) super.clone() ;
            HashMap<ParameterPath, AParameterDescription> newMap = new HashMap<>();
            for(AParameterDescription description : paramDescriptions.values()) {
                newMap.put(description.getPath(), description.clone()) ;
            }
            res.setParamDescriptions(newMap);
            //TODO: clone other members?
        } catch (CloneNotSupportedException e) {/*IGNORE*/ }
        return res ;
    }


//////////////////////////////
    ///////////////////// UTILITIES

    /**
     * add Parameter descriptions: this is a public facade to the "real" method
     * that adds only objects of type <TT>AParameterDescription</TT>
     * So if objects are not ot this actual type new objects of the needed type are created
     * (see <TT>AParameterDescription</TT> constructor documentation)
     * @param descriptions
     */
    @Override
    public void addParameterDescriptions(ParameterDescription... descriptions) {
        addParameterDescriptions(buildSafeArray(descriptions));
    }

    @Override
    public void addParameterDescriptions(Collection<ParameterDescription> descriptions) {
        for(ParameterDescription parm : descriptions) {
            if(! (parm instanceof AParameterDescription)) {
                this.addParameterDescriptions(new AParameterDescription(parm));
            } else {
                this.addParameterDescriptions((AParameterDescription) parm);
            }
        }
    }
    
    private AParameterDescription[] buildSafeArray(ParameterDescription... descriptions){
        List<ParameterDescription> list = Arrays.asList(descriptions);
        AParameterDescription[] parms = new AParameterDescription[descriptions.length] ;
        int ix = 0 ;
        for(ParameterDescription description: list) {
            if(! (description instanceof AParameterDescription)){
                parms[ix] = new AParameterDescription(description)  ;

            } else {
                parms[ix] = (AParameterDescription) description ;
            }
            ix++ ;
        }
        return parms ;
    }

    /*
    //works as is dues to equality rules
    @Override
    public void removeParameterDescriptions(ParameterDescription... descriptions) {
        removeParameterDescriptions(buildSafeArray(descriptions));
    }
    */

    /**
     * adds a list of parameter descriptions
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public void addParameterDescriptions(AParameterDescription... descriptions) {
        if(isReadOnly()) {
            throw new ImmutableStateException("parameter description list") ;
        }
        for(AParameterDescription description : descriptions) {
            this.paramDescriptions.put(description.getPath(), description) ;
            categorySet.add(description.getCategory());
        }
    }
    /**
     * removes a list of parameter descriptions.
     * <P>
     *     works with any subclass of parameterDescription (see equals method code)
     * </P>
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public void removeParameterDescriptions(ParameterDescription... descriptions) {
        if(isReadOnly()) {
            throw new ImmutableStateException("parameter description list") ;
        }
        for(ParameterDescription description : descriptions) {
            this.paramDescriptions.remove(description) ;
        }
    }

}
