package org.lsst.ccs.localdb.configdb.model;

import org.hibernate.annotations.Immutable;

import javax.persistence.*;
import java.util.*;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * An actual configuration profile for a given category.
 * Once registered instances of this class are immutable except for  comments
 * and registration of method calls (to be specified).
 *
 * @author bamade
 */
// Date: 12/04/12

@Table(uniqueConstraints = {
    @UniqueConstraint(columnNames = {"name", "tag", "subsystemName"})})
@Entity
@Immutable
class AConfigProfile extends ConfigProfile implements Cloneable {
    private static final long serialVersionUID = -6199141309488812631L;
    
    @Id
    @GeneratedValue
    private long id;

    @ManyToOne (fetch=FetchType.EAGER)
    private ASubsystemDescription subsystemDesc;

    /**
     * duplicate information : helps when in an intermediate state where
     * the subsystemDescription is being deprecated ;
     */
    private long subsystemId;
    
    //TODO:  annotate method instead to avoid bug with unModifiableConfigurations (the set was not invoked)
    @OneToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    private Set<AParameterConfiguration> parameterConfigurations =
            new HashSet<AParameterConfiguration>();
    
    AConfigProfile() {
    }

    /**
     * creates a config profile pointing to a Subsystem Description that must be registered in the database.
     *
     * @param subsystemDesc
     * @param name
     * @param userName
     * @throws IllegalArgumentException if the subsystem description is not registered or is deprecated
     */
    AConfigProfile(ASubsystemDescription subsystemDesc, String categoryName, String configName) {
        super(subsystemDesc.getSubsystemName(), subsystemDesc.getTag(), categoryName, configName);
        this.subsystemDesc = subsystemDesc;
        this.subsystemId = subsystemDesc.getId();
//        if (subsystemId == 0) {
//            throw new IllegalArgumentException("Subsystem description should have been registered in database");
//        }
        //TODO: tautological all instances of ASubsystemDescription have a STILL_VALID date!
        long endDate = subsystemDesc.getEndTimestamp();
        // when in remote operation this tet may not suffice
        if (!(endDate == PackCst.STILL_VALID)) {
            throw new IllegalArgumentException("using deprecated subsystem");
        }
    }


    /**
     * creates a copy of an active ConfigProfile (for instance to start an engineering mode or to validate
     * a new tagged configuration from an engineering mode configuration).
     *
     * @param anotherConfig
     * @param name
     * @param userName
     * @param level
     */
    AConfigProfile(AConfigProfile anotherConfig, String newName) {
        this(anotherConfig.getSubsystemDesc(), anotherConfig.getCategoryName(), 
                newName == null ? anotherConfig.getConfigName() : newName);

        for (AParameterConfiguration parmConfig : anotherConfig.getParameterConfigurations()) {
            // TODO : this code should be moved to ParameterConfiguration
            AParameterConfiguration copyParm = new AParameterConfiguration(parmConfig);
            this.addParameterConfigurations(copyParm);
        }
    }
    
    /////////////////////////////////// ACCESSORS/MUTATORS
    @Override
    public long getId() {
        return id;
    }
    
    @Override
    protected void setId(long id) {
        this.id = id;
    }


    @Override
    public ASubsystemDescription getSubsystemDescription() {
        return subsystemDesc;
    }

    /**
     * return <TT>getParameterConfigurations</TT>
     *
     * @return
     */
    @Override
    public Set<? extends ParameterConfiguration> getModifiedParameters() {
        return getParameterConfigurations();
    }
    
    ASubsystemDescription getSubsystemDesc() {
        return subsystemDesc;
    }

    void setSubsystemDesc(ASubsystemDescription subsystemDesc) {
        this.subsystemDesc = subsystemDesc;
    }

    /**
     * @return an unmodifiable set
     */
    public Set<AParameterConfiguration> getParameterConfigurations() {
        return Collections.unmodifiableSet(parameterConfigurations);
    }
    
    void setParameterConfigurations(Set<AParameterConfiguration> parameterConfigurations) {
        this.parameterConfigurations = parameterConfigurations;
    }

    public AConfigProfile clone() throws CloneNotSupportedException {
        AConfigProfile res = null;
        res = (AConfigProfile) super.clone();
        HashSet<AParameterConfiguration> newSet = new HashSet<AParameterConfiguration>();
        for (AParameterConfiguration parmConfig : parameterConfigurations) {
            newSet.add(parmConfig.clone());
        }
        res.setParameterConfigurations(newSet);
        return res;
    }

    public long getSubsystemId() {
        return subsystemId;
    }

    void setSubsystemId(long subsystemId) {
        this.subsystemId = subsystemId;
    }

    /**
     * Static addition of a parameter configuration.
     * @param parameterName
     * @param value 
     */
    public void addParameterConfiguration(String parameterName, String value){
          ParameterDescription description = subsystemDesc.fetch(parameterName);
          if (description != null) {
              addParameterConfigurations(new AParameterConfiguration((AParameterDescription)description, value));
          } else {
              Logger.getLogger("org.lsst.ccs.config").warn("incoherent name for parameter : " + parameterName + " : ignored");
          }
    }
    /**
     * registers a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException  if the object is already registered in database
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     *                                  subsystem description
     */
    // TODO: check if Description not deprecated!
    public void addParameterConfigurations(AParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("no modification of registered data");
        }
        for (AParameterConfiguration parameterConfiguration : parameterConfigurations) {
            AParameterDescription description = parameterConfiguration.getParameterDescription();
            if (description.getCategory().equals(getCategoryName())){
                // check for deprecation
                this.parameterConfigurations.add(parameterConfiguration);
            } else {
                Logger.getLogger("org.lsst.ccs.config").warn("The property " 
                        + description.getParameterName() 
                        + " does not belong to the category specified by the property file name : " + getCategoryName());
            }
        }
    }

    /**
     * adds parameter configuration objects to this profile
     * @param parameterConfigurations
     */
    public void addParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        addParameterConfigurations(buildSafeArray(parameterConfigurations));
    }

    /**
     * Casts the array of parameterConfiguration into an array of AParameterConfiguration.
     * @param parameterConfigurations
     * @throws IllegalArgumentException if at least one of the argument is not 
     * of instance AParameterConfiguration.
     * @return 
     */
    private AParameterConfiguration[] buildSafeArray(ParameterConfiguration... parameterConfigurations) {
        AParameterConfiguration[] parms = new AParameterConfiguration[parameterConfigurations.length];
        for (int ix = 0; ix < parameterConfigurations.length; ix++) {
            ParameterConfiguration conf = parameterConfigurations[ix];
            if (!(conf instanceof AParameterConfiguration)) {
                throw new IllegalArgumentException("deprecated parameter Configuration");
            }
            parms[ix] = (AParameterConfiguration) conf;

        }
        return parms;
    }

    /**
     * removes a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException if operated on an object registered to the database
     */
    public void removeParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Parameter Configuration list");
        }
        //todo: problem how to remove property "changingStaticData"? should we check all other parmConfigs?
        for (ParameterConfiguration parameterConfiguration : parameterConfigurations) {
            this.parameterConfigurations.remove(parameterConfiguration);
        }
    }
}
