package org.lsst.ccs.utilities.readout;

import org.lsst.ccs.utilities.ccd.CCDType;

/**
 * Represents readout parameters derived from the CCD geometry plus meta-data
 * registers read from the DAQ.
 *
 * @see ReadOutParametersBuilder
 * @author tonyj
 */
public abstract class ReadOutParameters {

    private final int[] registers;
    private final CCDType ccdType;
    private final ReadOutRegisterMap registerMap;

    ReadOutParameters(CCDType ccdType, int[] registers, ReadOutRegisterMap registerMap) {
        this.registers = registers;
        this.ccdType = ccdType;
        this.registerMap = registerMap;
    }

    public CCDType getCCDType() {
        return ccdType;
    }

    final public int getValue(String parameterName) {
        return registers[registerMap.getIndex(parameterName)];
    }

    /**
     * Utility method to get the the number of pixels read in the serial
     * direction. This is the sum of the read and over scan pixels. Notice that
     * readCols includes pre scan pixels in full frame mode.
     *
     * @return the number of pixels read in the serial direction.
     */
    public abstract int getSerialReadPixels();

    /**
     * Utility method to get the the number of pixels read in the parallel
     * direction. This is the sum of the read and over scan pixels.
     *
     * @return the number of pixels read in the parallel direction.
     */
    public abstract int getParallelReadPixels();

    /**
     * Get the total parallel size in pixels. This is the sum of the parallel
     * over scan and the active sizes.
     *
     * @return The total serial size in pixels.
     */
    public abstract int getTotalParallelSize();

    /**
     * Get the total parallel size in pixels. This is the sum of the parallel
     * over scan and the active sizes.
     *
     * @return The total serial size in pixels.
     */
    public abstract int getTotalSerialSize();

    /**
     * Get the serial prescan size. This corresponds to Designator variable
     * "preh" in Table 8 of LCA-13501
     *
     * The difference between the serial prescan size and the preCols quantity
     * is that the serial prescan is read in prescan mode, while the preCols are
     * skipped.
     *
     * @return The serial prescan size.
     */
    public abstract int getSerialPrescan();

    abstract void validate();

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        for (String name : registerMap.getNames()) {
            builder.append(name).append("=").append(getValue(name)).append(";");
        }
        return builder.toString();
    }
}
