package org.lsst.ccs.utilities.readout;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.MetaDataSet;

/**
 * An implementation of FitsHeaderMetadataProvider based on a Geometry object.
 * It internally creates CCD level FitsHeaderMetadataProvider objects that will
 * be used when fits files are written.
 * 
 * This is achieve by holding a reference to the Geometry object for which the
 * fits files are being written. The CCD information is extracted using the
 * ImageSet.getImageSetId() method that should return the unique CCD id  within
 * the provided geometry.
 *
 * @author The LSST CCS Team
 */
public class GeometryFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

    private final CCD ccd;
    
    private WcsCoordinatesFitsHeaderMetadataProvider provider;
    
    public GeometryFitsHeaderMetadataProvider(CCD ccd) {
        provider = new WcsCoordinatesFitsHeaderMetadataProvider(ccd);
       this.ccd = ccd;
    }
    @Override
    public MetaDataSet getPrimaryHeaderMetadata() {
        MetaDataSet mds = provider.getPrimaryHeaderMetadata();
        mds.addMetaData("", "CCDModel", ccd.getType().getType());
        mds.addMetaData("", "CCDManufacturer", ccd.getType().getManufacturer().toUpperCase());
        return mds;
    }

    @Override
    public MetaDataSet getDataExtendedHeaderMetadata(String imageExtensionName) {

        Segment segment = ccd.getSegmentByName(imageExtensionName);
        Map<String, Object> imageMetaData = new HashMap<>();
        
        imageMetaData.put("EXTNAME", imageExtensionName);
        imageMetaData.put("CHANNEL", segment.getChannel());
        imageMetaData.put("CCDSUM", "1 1");

        MetaDataSet metadata = provider.getDataExtendedHeaderMetadata(imageExtensionName);
        metadata.addMetaDataMap("channel", imageMetaData);
        return metadata;
    }

    @Override
    public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
        return provider.getAdditionalExtendedHeaderMetadata(extendedKeyword);
    }
    
    private static int getSegmentParallelPosition(Segment seg) {
        return seg.getParallelPosition();
    }
    private static int getSegmentSerialPosition(Segment seg) {
        return seg.getSerialPosition();
    }

}
