package org.lsst.ccs.utilities.image;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import nom.tam.fits.FitsDate;
import nom.tam.fits.FitsException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * A MetaDataSet is used to provide values for the fits header keywords.
 * Since there can be multiple sources of header keywords, this class provides
 * methods for adding key-value maps for a given source. The key corresponds
 * to the header keyword and the value represents it value. The source of origin
 * is used when resolving expressions for header keywords as described in spec
 * files as: ${metaMap.metaName}. "metaMap" corresponds to the source and "metaName"
 * to a key as coming from that specific source. The corresponding value is used
 * when writing out fits files.
 * 
 * Internally it uses a Map of Maps to store the meta-data.
 *
 * @author The LSST CCS Team.
 */
public class MetaDataSet {


    private final Map<String,List<Map<String,Object>>> metaDataMaps = new HashMap<>();

    /**
     * Add a set of properties to a MetaDataSet. The properties will be
     * converted to the appropriate object type based on their value.
     *
     * @param name The name of the map to which the properties should be added
     * @param props The properties.
     */
    public void addProperties(String name, Properties props) {
        addMetaDataMap(name, convertToMetaData(props));
    }

    
    /**
     * Add a meta-data entry to a named meta-data object. The named object will
     * be created if it does not already exist.
     *
     * @param name The name of the map to which the new meta-data should be
     * added
     * @param key The key of the meta-data to add
     * @param value The value of the meta-data to add
     */
    public void addMetaData(String name, String key, Object value) {
        List<Map<String, Object>> meta = findOrCreateMetaData(name);
        meta.add(0,Collections.singletonMap(key, value));
    }
    
    /**
     * Add a set of meta-data to a named meta-data object. The named object will
     * be created if it does not already exist.
     *
     * @param name The name of the map to which the new meta-data should be
     * added
     * @param data The meta-data to add
     */
    public void addMetaDataMap(String name, Map<String, Object> data) {
        if ( data == null ) {
            return;
        }
        List<Map<String, Object>> meta = findOrCreateMetaData(name);
        meta.add(0, data);
    }

    /**
     * Add a MetaDataSet to an existing MetaDataSet.
     * A named set will be created if it does not exist. If it already exists
     * the content of the new MetaDataSet will be added to the existing one.
     * 
     * @param metaDataSet The MetaDataSet to add to the existing one.
     */
    public void addMetaDataSet(MetaDataSet metaDataSet) {
        if ( metaDataSet == null ) {
            return;
        }
        for ( Entry<String,List<Map<String,Object>>> entry : metaDataSet.metaDataMaps.entrySet() ) {
            for (Map<String,Object> metaData : entry.getValue()) {
                addMetaDataMap(entry.getKey(), metaData);
            }
        }
        
    }
    
    /**
     * Convert a MetaDataSet to a Properties object.
     * Each entry is added with both with its full name (mapName.keyName)
     * and with its keyName alone.
     * 
     * @return The Properties object with the valued in this MetaDataSet.
     */
    public Properties convertToProperties() {
        // TODO: Only used by ImageProc?
        // MAX: yes, it's used only in ImageProc.
        throw new UnsupportedOperationException("convertToProperties");
    }
    
    public Object getValue(String name) {
        return getValue(null,name);
    }

    public Object getValue(String map, String name) {
        Map<String,Object> results = new HashMap<>(); 
        Object obj = null;
        for (Entry<String,List<Map<String, Object>>> metaDataEntry : metaDataMaps.entrySet()) {
            for (Map<String, Object> metaData : metaDataEntry.getValue()) {
                if ( map == null || map.equals(metaDataEntry.getKey())) {
                    Object result = metaData.get(name);
                    if (result != null) {
                        results.put(metaDataEntry.getKey(),result);
                        obj = result;
                    }
                }
            }
        }
        if ( results.size() <= 1 ) {
            return obj;
        } else {
            throw new RuntimeException("More than one map defines a value for "+name+" ("+results+")");
        }
    }

    private List<Map<String, Object>> findOrCreateMetaData(String name) {
        List<Map<String, Object>> result = metaDataMaps.get(name);
        if (result == null) {
            result = new LinkedList();
            metaDataMaps.put(name, result);
        }
        return result;
    }

    private Map<String, Object> convertToMetaData(Properties props) {
        Map<String, Object> result = new HashMap<>();
        // Note, this loops over all properties, including thos specified in the
        // default property list of the Properties object
        for (Object n : BootstrapResourceUtils.getAllKeysInProperties(props)) {            
            String name = (String)n;
            String value = props.getProperty(name);
            result.put(name, convertToMetaData(value));
        }
        return result;
    }

    private Object convertToMetaData(String value) {
        try {
            return Integer.decode(value);
        } catch (NumberFormatException x) {
            try {
                return Double.valueOf(value);
            } catch (NumberFormatException xx) {
                try {
                    return new FitsDate(value).toDate();
                } catch (FitsException xxx) {
                    if ("true".equalsIgnoreCase(value)) {
                        return Boolean.TRUE;
                    } else if ("false".equalsIgnoreCase(value)) {
                        return Boolean.FALSE;
                    } else {
                        return value;
                    }
                }
            }
        }
    }
}
