package org.lsst.ccs.utilities.image;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.HeaderCard;
import nom.tam.fits.HeaderCardException;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Date;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Float;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Integer;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.MJD;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class HeaderWriter {

    /**
     * Fill in a FITS header based on a set of HeaderSpecification.
     *
     * @param file The file being written (used for error messages)
     * @param hdu The HDU into which the header should be written
     * @param spec The header specifier to use
     * @param metaDataSet The meta-data to use to fill the header
     * @throws HeaderCardException
     * @throws IOException
     */
    public static void addMetaDataToHeader(File file, BasicHDU hdu, HeaderSpecification spec, MetaDataSet metaDataSet) throws HeaderCardException, IOException {
        for (HeaderSpecification.GroupLine group : spec.getGroups()) {
            Collection<HeaderSpecification.HeaderLine> headersForGroup = spec.getHeadersForGroup(group.getGroupName());
            if (!headersForGroup.isEmpty() && !group.getGroupName().isEmpty()) {
                for (String comment : group.getComments()) {
                    hdu.getHeader().insertCommentStyle("        ", comment);
                }
            }
            for (HeaderSpecification.HeaderLine header : headersForGroup) {

                Object value = header.getValue(metaDataSet);
                if (header.isRequired() && value == null) {
                    throw new FitsFileWriter.FitsIOException(file, "Could not find value for required header keyword: %s defined as: \n%s", header.getKeyword(), header.getHeaderDefinition());
                }
                try {
                    if (value != null) {
                        if (value.equals(HeaderSpecification.NULL)) {
                            hdu.getHeader().addLine(new HeaderCard(header.getKeyword(), null, header.getComment(), true));
                        } else {
                            switch (header.getDataType()) {
                                case Integer -> hdu.addValue(header.getKeyword(), ((Number) value).intValue(), header.getComment());
                                case Float -> {
                                    double data = ((Number) value).doubleValue();
                                    if (!Double.isFinite(data)) {
                                        throw new IllegalArgumentException("Can not store non-finite floating point in FITS file");
                                    }
                                    //The following code is to reduce the number of significant digits to 5
                                    //See https://jira.slac.stanford.edu/browse/LSSTCCS-2168
                                    hdu.addValue(header.getKeyword(), formatDouble(data,header), header.getComment());
                                }
                                case Boolean -> hdu.addValue(header.getKeyword(), (Boolean) value, header.getComment());
                                case Date, DateTAI -> hdu.addValue(header.getKeyword(), DateUtils.convertTemporalToString(value), header.getComment());
                                case DateUTC -> { // Only supported for CCSTimeStamps
                                    if (value instanceof CCSTimeStamp timestamp) {
                                        hdu.addValue(header.getKeyword(), DateUtils.convertTemporalToString(timestamp.getUTCInstant()), header.getComment());
                                    } else {
                                        throw new IllegalArgumentException("DateUTC only supported for CCSTimeStamp");
                                    }
                                }
                                case MJD, MJDTAI -> hdu.addValue(header.getKeyword(), DateUtils.convertTemporalToMJD(value), header.getComment());
                                case MJDUTC -> { // Only supported for CCSTimeStamps
                                    if (value instanceof CCSTimeStamp timestamp) {
                                        hdu.addValue(header.getKeyword(), DateUtils.convertTemporalToMJD(timestamp.getUTCInstant()), header.getComment());                            
                                    } else {
                                        throw new IllegalArgumentException("MJDUTC only supported for CCSTimeStamp");
                                    }
                                }
                                default -> hdu.addValue(header.getKeyword(), String.valueOf(value), header.getComment());
                            }
                        }
                    }
                } catch (ClassCastException x) {
                    throw new FitsFileWriter.FitsIOException(file, x, "Meta-data header %s with value %s(%s) cannot be converted to type %s", header.getKeyword(), value, value.getClass(), header.getDataType());
                } catch (IllegalArgumentException | HeaderCardException x) {
                    throw new FitsFileWriter.FitsIOException(file, x, "Meta-data header %s with value %s(%s) cannot be written", header.getKeyword(), value, value.getClass());
                }
            }
        }
    }

    public static double formatDouble(double d, HeaderLine header) {
        String format = header.getFormat();
        String dStr = String.format(format.isBlank()? "%5g": format, d);
        return Double.parseDouble(dStr);
    }

}
