package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.awt.Point;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Base Geometry: a rectangle inside which it is possible to add other
 * Geometries.
 *
 * Note on coordinate system and dimensions.
 * The (x,y) coordinate system refers to Java Image/Component
 * convention. (0,0) is the upper left corner with x increasing to the right
 * and y increasing downward. For simplicity (x,y) cannot be negative.
 * The "width" of a Geometry is its maximum extent along the x direction and the
 * "height" is along the y direction.
 * 
 * Since this class is designed to describe Camera components inside the Focal Plane
 * (Rafts, Rebs, CCDs, Segments) we also support a coordinate system to numerically 
 * describe the position of a component inside its parent.
 * For uniformity with the Camera Coordinate System we will call this coordinate
 * system parallel vs serial.
 *
 * This coordinate system is used to count components within a parent. It has nothing
 * to do with dimensions.
 * The parallel coordinate is parallel to the x direction growing to the left.
 * The serial coordinate is parallel to the y direction growing upward.
 * The origin of the (p,s) coordinate system is at the bottom right corner of 
 * the Geometry object.
 * 
 * So for a parent containing 4 children in the x direction and 3 in the y direction
 * the children will be represented as follows:
 *  - child (0,0) is at the bottom right corner
 *  - child (0,2) is at the top right corner
 *  - child (3,0) is at the bottom left corner
 *  - child (3,2) is at the top left corner
 * 
 * This class is designed to contain a rectangular pattern of homogenous children.
 * Not all slots have to be filled with children.
 * 
 * This class is not thread safe.
 *
 * @author The LSST CCS Team
 * 
 * @param <T> The template of the classes that can be added to this Geometry
 */
public abstract class Geometry<T extends Geometry> {

    private Dimension dimension;
    private final List<InnerGeometry<T>> components = new ArrayList<>();
    private Geometry parent;
    private boolean hasFixedDimension = false;
    protected String name;
    private final Geometry[][] children; 
    private final int serialChildrenCount, parallelChildrenCount;
    private int serialPosition = 0, parallelPosition = 0;

    /**
     * Create a new Geometry with a maximum set of components in the (p,s) 
     * coordinate system.
     * 
     * Subclasses of Geometries that allow children to be added must override
     * method {@link #addGeometryToGrid(org.lsst.ccs.geometry.Geometry, int, int) addGeometryToGrid}.
     * 
     * @param name The name of this Geometry
     * @param dimension The dimension of this Geometry. Children must fit within this dimension.
     * @param p The number of allowed components in the parallel direction.
     * @param s The number of allowed components in the serial direction
     */
    public Geometry(String name, Dimension dimension, int p, int s) {
        this.dimension = dimension;
        this.hasFixedDimension = true;
        this.name = name;
        children = ( p > 0 && s > 0 ) ? new Geometry[p][s] : null;
        this.serialChildrenCount = s;
        this.parallelChildrenCount = p;
    }

    /**
     * Create a new Geometry with no children.
     * If an attempt is made to add children an exception will be thrown.
     * 
     * @param name The name of this Geometry
     * @param dimension The dimension of this Geometry.
     */
    public Geometry(String name, Dimension dimension) {
        this(name, dimension, 0, 0);
    }
    
    /**
     * Get the Geometry name
     * @return The Geometry name.
     */
    public String getName() {
        return name+parallelPosition+serialPosition;
    }
    
    /**
     * Get the Geometry width.
     *
     * @return The width of the Geometry;
     */
    public int getWidth() {
        return dimension.width;
    }

    /**
     * Get the Geometry height;
     *
     * @return The height of the Geometry;
     */
    public int getHeight() {
        return dimension.height;
    }

    /**
     * Get the Dimension of this Geometry.
     *
     * @return The Dimension of this Geometry.
     */
    public Dimension getDimension() {
        return new Dimension(dimension);
    }

    /**
     * Get the parent of this Geometry. Null will be returned if this is an
     * orphan.
     *
     * @return The parent (null if this is an orphan).
     */
    public Geometry getParent() {
        return parent;
    }
    
    /**
     * Find the closest containing Geometry of a given type.
     * 
     * @param clazz The type of the parent to be found
     * @return      The closest enclosing geometry for the given type.
     */
    public Geometry<?> findParentOfType(Class clazz) {
        Geometry local_parent = getParent();
        if ( local_parent == null ) {
            return null;
        }
        if ( clazz.isInstance(local_parent) ) {
            return local_parent;
        }
        return local_parent.findParentOfType(clazz);
    }    
    
    /**
     * Check if this Geometry has a Parent.
     *
     * @return true if this Geometry has a parent.
     *
     */
    public boolean hasParent() {
        return getParent() != null;
    }

    protected final void setParent(Geometry parent) {
        if (this.parent != null) {
            throw new RuntimeException("This Geometry already already has a parent!");
        }
        this.parent = parent;
    }

    /**
     * Add a Geometry to this Geometry at a given (x,y) location.
     * This method is protected because it should only be used internally by
     * classes extending the Geometry class.
     * 
     * Users will have to use the {@link #addChildGeometry(T, int, int) addChildGeometry}
     * method to place a geometry in the (p,s) coordinate system.
     * 
     * The convention is the the new Geometry upper left corner will be placed at (x,y).
     * Child Geometries cannot be added to a Geometry that already has a parent AND
     * does not have a fixed dimension. The reason for this choice is to simplify
     * the logic of boundaries breaking.
     * 
     * Coordinate x and y must be positive otherwise an exception will be thrown.
     * 
     * If the this parent Geometry has fixed boundaries an exception will be thrown if the
     * child Geometry to be added does not fit inside its boundaries.
     * 
     * An exception will be thrown if the child Geometry being added already
     * belongs to another Geometry.
     *
     * @param child The Geometry to be added
     * @param x The x coordinate of the child Geometry location
     * @param y The y coordinate of the child Geometry location
     */
    protected void addGeometry(T child, int x, int y) {
        if (getParent() != null && !hasFixedDimension) {
            throw new RuntimeException("Cannot add a child Geometry to a non orfan parent Geometry"
                    + " with variable dimensions");
        }
        if (child.getParent() != null) {
            throw new RuntimeException("The child geometry already belongs to another Geometry.");
        }
        if (x < 0 || y < 0) {
            throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), coordiantes cannot be negative");
        }
        if (hasFixedDimension) {
            if (x > getWidth() || y > getHeight()) {
                throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), it is outside of this geometry's size " + getWidth() + "x" + getHeight());
            }
            if (x + child.getWidth() > getWidth() || y + child.getHeight() > getHeight()) {
                throw new IllegalArgumentException("The child Geometry cannot be added within the boundaries of this Geometry " + getWidth() + "x" + getHeight());
            }
        }

        child.setParent(this);
        Point point = new Point(x, y);
        components.add(new InnerGeometry<>(child, point));
        if (!hasFixedDimension) {
            dimension.setSize(Math.max(getWidth(), x + child.getWidth()), Math.max(getHeight(), y + child.getHeight()));
        }
    }

    /**
     * Replace a Geometry with a new one in the same place.
     * @param oldChild The existing child.
     * @param newChild The new child.
     */
    protected void replaceExistingGeometry(T oldChild, T newChild) {
        if ( ! oldChild.hasParent() ) {
            throw new RuntimeException("Cannot replace a geometry that does not have a parent");
        }
        if ( oldChild.getParent() != this ) {
            oldChild.getParent().replaceExistingGeometry(oldChild, newChild);
        } else {
            if (newChild.getClass() != oldChild.getClass()) {
                throw new RuntimeException("Cannot replace a geometry with a different type");
            }
            for (Iterator<InnerGeometry<T>> iter = components.iterator(); iter.hasNext();) {
                InnerGeometry<T> innerGeom = iter.next();
                if (innerGeom.getGeometry() == oldChild) {
                    iter.remove();
                    break;
                }
            }
            addChildGeometry(newChild, oldChild.getParallelPosition(), oldChild.getSerialPosition());
        }
    }

    /**
     * The maximum number of children in the serial direction.
     * 
     * @return The maximum number of children in the serial direction.
     */
    public int getSerialChildrenCount() {
        return this.serialChildrenCount;
    }

    /**
     * The maximum number of children in the parallel direction.
     * 
     * @return The maximum number of children in the parallel direction.
     */
    public int getParallelChildrenCount() {
        return this.parallelChildrenCount;
    }
    
    /**
     * The serial position of this Geometry within its parent.
     * 
     * @return The serial position of this Geometry within its parent.
     */
    public int getSerialPosition() {
        return this.serialPosition;
    }

    /**
     * The parallel position of this Geometry within its parent.
     * 
     * @return The parallel position of this Geometry within its parent.
     */
    public int getParallelPosition() {
        return this.parallelPosition;
    }
    
    protected void setSerialPosition(int serialPosition) {
        this.serialPosition = serialPosition;
    }

    protected void setParallelPosition(int parallelPosition) {
        this.parallelPosition = parallelPosition;
    }
    
    /**
     * Add a child to the grid to a given position in the parallel,serial coordinate
     * system.
     * 
     * @param child The child to be added
     * @param s     The serial position to add it to 
     * @param p     The parallel position to add it to
     */    
    public void addChildGeometry(T child, int p, int s) {
        if ( children == null ) {
            throw new RuntimeException("This geometry does not allow children.");
        }
        children[p][s] = child;
        child.setParallelPosition(p);
        child.setSerialPosition(s);
        addGeometryToGrid(child, p, s);
    }
    
    /**
     * This method must be implemented by all classes extending Geometry.
     * It should internally convert the (p,s) coordinate for the child to its
     * (x,y) position within this geometry.
     * 
     * @param child The child to be added.
     * @param p     The parallel position
     * @param s     The serial position.
     */
    protected abstract void addGeometryToGrid(T child, int p, int s);
    
    /**
     * Get child at position (p,s).
     * @param p     The parallel position to add it to
     * @param s     The serial position to add it to 
     * @return      The child at the given position.
     * 
     */
    public T getChild(int p, int s) {
        return (T)children[p][s];
    }
    
    /**
     * Check if this Geometry has children.
     *
     * @return true if this Geometry has children.
     */
    public boolean hasChildren() {
        return !components.isEmpty();
    }

    /**
     * Get a map for all the children contained in this Geometry with the
     * corresponding Point which represent the location
     * of the child Geometry in the topmost geometry.
     *
     * @return A map of Geometry, Point
     */
    public final Map<Geometry, Point> getChildrenWithAbsoluteCoordinates() {
        HashMap<Geometry, Point> map = new HashMap<>();
        for (InnerGeometry ig : components) {
            map.put(ig.getGeometry(), getAbsolutePoint(ig.getPoint()));
        }
        return map;
    }

    /**
     * Get the list of children for this Geometry.
     * The order of the children depends on how they were added to the geometry.
     *
     * @return A list of children Geometry objects
     */
    public final List<T> getChildrenList() {
        ArrayList<T> c = new ArrayList<>();
        for (InnerGeometry<T> ig : components) {
            c.add(ig.getGeometry());
        }
        return c;
    }

    /**
     * Get this Geometry unique id.
     * This is built by going over the parents and concatenating their name
     * separated by columns.
     * @return The Geometry unique id.
     * 
     */
    public String getUniqueId() {
        return buildUniqueId(this,"");        
    }
    
    private static String buildUniqueId(Geometry g, String name) {
        name = name.isEmpty() ? g.getName() : g.getName()+"."+name;
        if ( g.hasParent() ) {
            return buildUniqueId(g.getParent(), name);
        }
        return name;
    }
    
    /**
     * Find a Geometry by name.
     * 
     * @param geometryUniqueId The unique id of the geometry to be found.
     * @return The geometry or null if the geometry is not found
     * 
     */
    public final Geometry findGeometry(String geometryUniqueId) {
        return findGeometry(this, geometryUniqueId);        
    }
    
    private static Geometry findGeometry(Geometry<?> g, String geometryUniqueId) {
        if (g.getUniqueId().equals(geometryUniqueId)) {
            return g;
        }
        for ( Geometry child : g.getChildrenList() ) {
            Geometry found = findGeometry(child, geometryUniqueId);
            if ( found != null ) {
                return found;
            }
        }
        return null;
    }
    /**
     * Get the number of children added to this Geometry.
     *
     * @return the number of children in the geometry.
     */
    public final int getNumberOfChildren() {
        return components.size();
    }

    /**
     * Get the absolute location of this Geometry with respect to the topmost
     * geometry, that is the one without a parent.
     * 
     * @return The absolute location for this Geometry.
     */
    public Point getGeometryAbsolutePosition() {
        return getAbsolutePoint(new Point(0,0));
    }
    
    /**
     * Convert a point within this Geometry to the absolute coordinate system
     * of the topmost Geometry.
     * 
     * @param p The point to convert.
     * @return  The absolute Point representing the provided point.
     */
    public Point getAbsolutePoint(Point p) {
        if (!hasParent()) {
            return p;
        }
        Point thisGeometryPoint = getParent().getGeometryPoint(this);
        Point p1 = new Point(p.x+thisGeometryPoint.x, p.y+thisGeometryPoint.y);
        return getParent().getAbsolutePoint(p1);
    }

    /**
     * Get the location Point of a child Geometry within this Geometry.
     *
     * @param geometry The child Geometry.
     * @return The Point where the provided Geometry is attached to this Geometry.
     * 
     */
    public Point getGeometryPoint(Geometry geometry) {
        for (InnerGeometry ig : components) {
            if (ig.getGeometry() == geometry) {
                return ig.getPoint();
            }
        }
        throw new RuntimeException("Could not find location for this Geometry: " + geometry);
    }

    private class InnerGeometry<K extends Geometry> {

        private final K geometry;
        private final Point point;

        InnerGeometry(K geometry, Point p) {
            this.geometry = geometry;
            this.point = p;
        }

        K getGeometry() {
            return geometry;
        }

        int getXPosition() {
            return point.x;
        }

        int getYPosition() {
            return point.y;
        }

        Point getPoint() {
            return point;
        }

    }
    
}
