package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.List;

/**
 * A class that defines the type of a CCD.
 * A CCDType is defined by:
 *  - A name
 *  - The CCD geometry constants as defined in CCDGeometryConstants
 * 
 * Two subclasses are provided: E2VCCDType and ITLCCDType. Other instances
 * can be created when needed by a test-stand or testing environment by 
 * extending this class.
 * 
 * @author The LSST CCS Team
 */
public class CCDType {

    private final CCDGeometryConstants ccdGeometryConstants;
    private final String name, type;
    private static final List<CCDType> knownTypes = new ArrayList<>();
    
    static {
        knownTypes.add(new ITLCCDType());
        knownTypes.add(new E2VCCDType());
        knownTypes.add(new WFCCDType());
    }
    
    public CCDType(String name) {
        this(getCCDType(name));
    }
    
    protected CCDType(CCDType type) {
        this(type.getName(), type.getType(), type.getCCDGeometryConstants());
    }
    
    /**
     * Create a CCDType.
     * 
     * @param name The name of this CCDType
     * @param ccdGeometryConstants The geometry constants for this CCD.
     */
    protected CCDType(String name, String type, CCDGeometryConstants ccdGeometryConstants) {                
        this.ccdGeometryConstants = ccdGeometryConstants;
        this.name = name;
        this.type = type;
    }
    
    /**
     * Get the CCDGeometryConstants for this CCDType.
     * 
     * @return The CCD geometry constants for this type of CCD.
     */
    public CCDGeometryConstants getCCDGeometryConstants() {
        return ccdGeometryConstants;
    }
    
    public String getManufacturer() {
        return getName();
    }
    
    /**
     * The name of this CCDType.
     * 
     * @return The name for this type of CCD.
     */
    public String getName() {
        return name;
    }
    
    /**
     * The type of this CCDType.
     * 
     * @return The type for this type of CCD.
     */
    public String getType() {
        return type;
    }
    
    /**
     * Get a CCDType by its name.
     * 
     */
    public static CCDType getCCDType(String name) {
        for (CCDType type : knownTypes) {
            if ( type.getName().toLowerCase().equals(name.toLowerCase()) ) {
                return type;
            }
        }
        throw new IllegalArgumentException("No known CCDType is available with name \""+name+"\"");
    }
    
    /**
     * Register a CCDType so that it can be fetched by name with {@code getCCDType}.
     * 
     */
    public static void registerCCDType(CCDType type) {
        knownTypes.add(type);
    }

    @Override
    public int hashCode() {
        return type.hashCode() + 34 * name.hashCode() + 22 * ccdGeometryConstants.hashCode();
    }


    
    @Override
    public boolean equals(Object obj) {
        if ( obj == null || ! (obj instanceof CCDType ) ) {
            return false;
        }
        return ((CCDType) obj).getName().equals(getName());
    }

    @Override
    public String toString() {
        return getName();
    }
    
}


