package org.lsst.ccs.drivers.reb;

/**
 ***************************************
 *
 *  REB status block access routines.
 *
 *  @author Owen Saxton
 *
 ***************************************
 */
public class StatusSet {

   /**
    *  Public constants.
    */
    public static final int
        REG_STATUS_0       = 0x800000,
        REG_STATUS         = 0xa00000,
        OFF_STAT_VERSION   = 0,
        OFF_STAT_LINKSTAT  = 1,
        OFF_STAT_CERR_CNT  = 2,
        OFF_STAT_LDWN_CNT  = 3,
        OFF_STAT_LERR_CNT  = 4,
        OFF_STAT_BUFFSTAT  = 5,
        OFF_STAT_RX_CNTS   = 6,
        OFF_STAT_TX_CNTS   = 10,
        OFF_STAT_SENT_CNT  = 14,
        OFF_STAT_DISC_CNT  = 15,
        OFF_STAT_TRUNC_CNT = 16,
        OFF_STAT_FORMAT    = 17,
        NUM_STAT_REG       = 18;
 
   /**
    *  Private data.
    */
    BaseSet bss;


   /**
    *  Constructor.
    *
    *  @param  bss  The associated base set object
    */
    public StatusSet(BaseSet bss)
    {
        this.bss = bss;
    }




   /**
    *  Enables the status block.
    *
    *  @throws  REBException 
    */
    public void enable() throws REBException
    {
        bss.enable(BaseSet.RSET_STATUS);
    }


   /**
    *  Disables the status block.
    *
    *  @throws  REBException 
    */
    public void disable() throws REBException
    {
        bss.disable(BaseSet.RSET_STATUS);
    }


   /**
    *  Gets the status block trigger time.
    *
    *  @return  The Unix millisecond time of the last status block trigger
    *
    *  @throws  REBException 
    */
    public long getTriggerTime() throws REBException
    {
        return bss.getTriggerTime(BaseSet.RSET_STATUS);
    }


   /**
    *  Reads the status block.
    *
    *  @return  The contents of the status block, as a Status object
    *
    *  @throws  REBException 
    */
    public Status readStatus() throws REBException
    {
        Status status = new Status();
        int[] regs = new int[NUM_STAT_REG];
        bss.read(bss.getVersion(BaseSet.OPTN_STATUS) != BaseSet.VERSION_0
                   ? REG_STATUS : REG_STATUS_0, regs);
        status.version = regs[OFF_STAT_VERSION];
        status.linkStat = regs[OFF_STAT_LINKSTAT];
        status.cerrCount = regs[OFF_STAT_CERR_CNT];
        status.ldwnCount = regs[OFF_STAT_LDWN_CNT];
        status.lerrCount = regs[OFF_STAT_LERR_CNT];
        status.buffStat = regs[OFF_STAT_BUFFSTAT];
        for (int j = 0; j < status.rxCount.length; j++) {
            status.rxCount[j] = regs[OFF_STAT_RX_CNTS + j];
        }
        for (int j = 0; j < status.txCount.length; j++) {
            status.txCount[j] = regs[OFF_STAT_TX_CNTS + j];
        }
        status.sentCount = regs[OFF_STAT_SENT_CNT];
        status.discCount = regs[OFF_STAT_DISC_CNT];
        status.truncCount = regs[OFF_STAT_TRUNC_CNT];
        status.format = regs[OFF_STAT_FORMAT];

        return status;
    }

}
