package org.lsst.ccs.drivers.reb.sim;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.Image;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.sim.SequencerSimulation.StateListener;

/**
 * A simulation of the DAQv1 image client with a REB3
 *
 * @author tonyj
 */
public class ImageClientDaq1Simulation implements ImageClient.Impl {

    private final HandleAndIdManager manager;
    private REB3Simulation reb3;
    private final AutoCloseableReentrantLock stateLock = new AutoCloseableReentrantLock();
    private final Condition imageDelivered = stateLock.newCondition();
    private final StateListener stateListener = (SequencerSimulation.State oldState, SequencerSimulation.State newState) -> {
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            if (newState == SequencerSimulation.State.RUNNING) {
                tag = reb3.getTriggerTime(BaseSet.RSET_SEQUENCER);
            }
            // Note: For the moment we will assume that an Image is delivered every time the sequencer
            // transitions from RUNNNING to STOPPED. This is not really true, we will provide a better
            // implementation later
            if (oldState == SequencerSimulation.State.RUNNING && newState == SequencerSimulation.State.STOPPED) {
                length = 4456448*3*16; // FIXME:
                imageDelivered.signalAll();
            }
        }
    };
    private int schema;
    private int version;
    private int id;
    private long tag;
    private int length;
    private final DataProvider dataProvider;
    
    ImageClientDaq1Simulation(HandleAndIdManager manager, DataProvider provider) {
        this.manager = manager;
        this.dataProvider = provider;
    }

    @Override
    public void newImageClient(int id, String ifc) {
        // Note: The IFC is not actually used in this (or any?) implementation.
        // Note: My guess is that newImageClient is actually only called once 
        // for any instance of ImageClient.Impl, otherwise the other methods
        // of this class do not really make sense.
        AddressSpace rs = manager.getAddressSpaceForId(id);
        if (rs instanceof REB3Simulation) {
            if (reb3 != null) {
                throw new SimulationException("Unsupported repeat call to newImageClient");
            }
            reb3 = (REB3Simulation) rs;
            reb3.getSequencer().addStateListener(stateListener);
            schema = reb3.read(BaseSet.REG_SCHEMA);
            version = reb3.read(BaseSet.REG_VERSION);
            this.id = reb3.read(BaseSet.REG_ID);
        } else {
            throw new SimulationException("Unsupported implementation of AddressSpace returned");
        }
    }

    @Override
    public void deleteImageClient() {
        if (reb3 != null) {
            reb3.getSequencer().removeStateListener(stateListener);
        }
    }

    @Override
    public Image waitForImage(Image image) {
        SimulatedImage result = image instanceof SimulatedImage ? (SimulatedImage) image : new SimulatedImage(this);
        result.setSchema(schema);
        result.setVersion(version);
        result.setAddress(id);
        // Note: It is not clear if we should return immediately if an image has been delivered
        // already. For now we always wait.
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            imageDelivered.await();
            result.setLength(length);
            result.setTag(tag);
            return result;
        } catch (InterruptedException ex) {
            throw new SimulationException("Interrupted while waiting for image", ex);
        }
    }

    @Override
    public boolean getImage(Image image) {
        if (image instanceof SimulatedImage) {
            SimulatedImage simImage = (SimulatedImage) image;
            //When the image is produced we fetch the values of
            //the registers in which we stored the image metadata.
            //Is this the right place for it?
            int[] registers = manager.getRegisters();
            int[] metadata = null;
            if ( registers != null ) {            
                metadata = new int[registers.length];
                for (int i = 0; i < registers.length; i++) {
                    reb3.readRegs(registers[i], metadata, i, 1);
                }
            }
            simImage.setRegisters(metadata);
            simImage.setData(dataProvider.getData(image.getMetadata()));
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void reset() {
        // This should reset the reb3.
    }

    @Override
    public void deleteImageMetadataRef(Image image) {
        // Nothing to be done.
    }

}
