package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ******************************************************************************
 **
 **  Power ADCs reading routines.
 **
 **  There are four supported board versions:
 **
 **    VERSION_0: REB0
 **    VERSION_1: REB1+
 **    VERSION_2: WREB
 **    VERSION_3: GREB
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class PowerAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_POWER_ADCS   = 0x600000,
        NUM_POWER_REGS_S = 8,
        NUM_POWER_REGS_W = 12,
        NUM_POWER_REGS_G = 10,
        NUM_POWER_REGS_M = NUM_POWER_REGS_W,

        ADC_DIG_VOLTAGE  = 0,
        ADC_DIG_CURRENT  = 1,
        ADC_ANA_VOLTAGE  = 2,
        ADC_ANA_CURRENT  = 3,
        ADC_CLKH_VOLTAGE = 4,
        ADC_CLKH_CURRENT = 5,
        ADC_OD_VOLTAGE   = 6,
        ADC_OD_CURRENT   = 7,
        ADC_HTR_VOLTAGE  = 8,
        ADC_HTR_CURRENT  = 9,
        ADC_DPHI_VOLTAGE = 10,
        ADC_DPHI_CURRENT = 11,

        ERROR_MASK       = 0x00010000,
        VALUE_MASK       = 0x0000ffff;

    public final static double
        VOLTAGE_SCALE    = 1.0 / 640.0,
        CURRENT_SCALE_LO = 10.0 / 640000.0,
        CURRENT_SCALE_HI = 10.0 / 640000.0 / 3.0,
        POWER_ERROR      = -8888.0;
 
   /**
    ***************************************************************************
    **
    **  Private data
    **
    ***************************************************************************
    */
    private static final Map<Integer, Integer> chanMapS = new HashMap<>();
    static {
        chanMapS.put(ADC_DIG_VOLTAGE, 0);
        chanMapS.put(ADC_DIG_CURRENT, 1);
        chanMapS.put(ADC_ANA_VOLTAGE, 2);
        chanMapS.put(ADC_ANA_CURRENT, 3);
        chanMapS.put(ADC_CLKH_VOLTAGE, 4);
        chanMapS.put(ADC_CLKH_CURRENT, 5);
        chanMapS.put(ADC_OD_VOLTAGE, 6);
        chanMapS.put(ADC_OD_CURRENT, 7);
    }

    private static final Map<Integer, Integer> chanMapW = new HashMap<>();
    static {
        chanMapW.put(ADC_DIG_VOLTAGE, 0);
        chanMapW.put(ADC_DIG_CURRENT, 1);
        chanMapW.put(ADC_ANA_VOLTAGE, 8);
        chanMapW.put(ADC_ANA_CURRENT, 9);
        chanMapW.put(ADC_CLKH_VOLTAGE, 2);
        chanMapW.put(ADC_CLKH_CURRENT, 3);
        chanMapW.put(ADC_OD_VOLTAGE, 10);
        chanMapW.put(ADC_OD_CURRENT, 11);
        chanMapW.put(ADC_HTR_VOLTAGE, 6);
        chanMapW.put(ADC_HTR_CURRENT, 7);
        chanMapW.put(ADC_DPHI_VOLTAGE, 4);
        chanMapW.put(ADC_DPHI_CURRENT, 5);
    }

    private static final Map<Integer, Integer> chanMapG = new HashMap<>();
    static {
        chanMapG.put(ADC_DIG_VOLTAGE, 0);
        chanMapG.put(ADC_DIG_CURRENT, 1);
        chanMapG.put(ADC_ANA_VOLTAGE, 6);
        chanMapG.put(ADC_ANA_CURRENT, 7);
        chanMapG.put(ADC_CLKH_VOLTAGE, 2);
        chanMapG.put(ADC_CLKH_CURRENT, 3);
        chanMapG.put(ADC_OD_VOLTAGE, 8);
        chanMapG.put(ADC_OD_CURRENT, 9);
        chanMapG.put(ADC_HTR_VOLTAGE, 4);
        chanMapG.put(ADC_HTR_CURRENT, 5);
    }

    private static final Map<Integer, Map> chanMapMap = new HashMap<>();
    static {
        chanMapMap.put(VERSION_0, chanMapS);
        chanMapMap.put(VERSION_1, chanMapS);
        chanMapMap.put(VERSION_2, chanMapW);
        chanMapMap.put(VERSION_3, chanMapG);
    }

    private static final Map<Integer, Integer> numRegsMap = new HashMap<>();
    static {
        numRegsMap.put(VERSION_0, NUM_POWER_REGS_S);
        numRegsMap.put(VERSION_1, NUM_POWER_REGS_S);
        numRegsMap.put(VERSION_2, NUM_POWER_REGS_W);
        numRegsMap.put(VERSION_3, NUM_POWER_REGS_G);
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public PowerAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public PowerAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the power ADC reading.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws REBException
    {
        getVersion(OPTN_BOARD_POWER);
        enable(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void waitDone() throws REBException
    {
        getVersion(OPTN_BOARD_POWER);
        waitDone(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable.
    **
    **  @return  The Unix millisecond time of the read enable
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws REBException
    {
        checkNotVersion(OPTN_BOARD_POWER, VERSION_0);
        return getTriggerTime(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one power value.
    **
    **  @param  adc  The number of the ADC to read
    **
    **  @return  The ADC value (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readAdc(int adc) throws REBException
    {
        Map<Integer, Integer> chanMap = getChanMap();
        Integer regNum = chanMap.get(adc);
        if (regNum == null) {
            throw new REBException("Invalid power ADC number");
        }
        enable();
        waitDone();
        int value = read(REG_POWER_ADCS + regNum);

        return (value & ERROR_MASK) != 0
                 ? POWER_ERROR : getScale(adc) * (value & VALUE_MASK);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the power values.
    **
    **  @return  The array of converted ADC values
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of power values.
    **
    **  @param  first  The number of the first ADC to read
    **
    **  @param  count  The number of ADCs to read
    **
    **  @return  The array of converted ADC values
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid power ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[getNumRegs()];
        double[] data = new double[count];
        read(REG_POWER_ADCS, rawData);
        Map<Integer, Integer> chanMap = getChanMap();
        for (int j = 0; j < count; j++) {
            int adc = first + j;
            int value = rawData[chanMap.get(adc)];
            data[j] = (value & ERROR_MASK) != 0
                        ? POWER_ERROR : getScale(adc) * (value & VALUE_MASK);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ADC registers.
    **
    **  @return  The number of ADC registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumRegs() throws REBException
    {
        return numRegsMap.get(getVersion(OPTN_BOARD_POWER));
    }


   /**
    ***************************************************************************
    **
    **  Gets the ADC channel map.
    **
    **  @return  The map of ADC channels to register offsets
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private Map<Integer, Integer> getChanMap() throws REBException
    {
        return chanMapMap.get(getVersion(OPTN_BOARD_POWER));
    }


   /**
    ***************************************************************************
    **
    **  Gets the scale factor.
    **
    **  @param  adc  The ADC number
    **
    **  @return  The scale factor for the measured quantity
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private double getScale(int adc) throws REBException
    {
        if ((adc & 1) == 0) {
            return VOLTAGE_SCALE;
        }
        else if (getVersion(OPTN_BOARD_POWER) != VERSION_2) {
            return adc <= ADC_ANA_CURRENT ? CURRENT_SCALE_LO : CURRENT_SCALE_HI;
        }
        else {
            return CURRENT_SCALE_LO;
        }
    }

}
