package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  ASIC ADC reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class FastAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_FADC_COUNT    = 0xc00000,
        REG_FADC_SAMPS    = 0xc00010,
        NUM_FADC_SAMPS    = 16;

    public final static double
        FADC_SCALE      = 1.0 / 2048;
 

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public FastAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public FastAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the fast ADC reading.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void enableFast() throws REBException
    {
        getVersion(OPTN_FAST_ADCS);
        enable(RSET_FAST_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the fast ADC read to finish.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void waitFastDone() throws REBException
    {
        getVersion(OPTN_FAST_ADCS);
        waitDone(RSET_FAST_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Sets the number of fast ADC samples.
    **
    **  @param  count  The number of samples to read
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setFastCount(int count) throws REBException
    {
        getVersion(OPTN_FAST_ADCS);
        write(REG_FADC_COUNT, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets the last few fast ADC samples.
    **
    **  @param  count  The number of samples to read
    **
    **  @return  A two-element array of sample value arrays, one per ADC
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[][] getFast(int count) throws REBException
    {
        getVersion(OPTN_FAST_ADCS);
        checkFastCount(count);
        double[][] value = new double[2][count];
        double[] sValue = new double[2];
        for (int j = 0; j < count; j++) {
            readFastSample(j, sValue);
            value[0][j] = sValue[0];
            value[1][j] = sValue[1];
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Gets the average of the last few fast ADC samples.
    **
    **  @param  count  The number of samples to read
    **
    **  @return  A two-element array of averaged samples, one per ADC
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] getFastAvge(int count) throws REBException
    {
        getVersion(OPTN_FAST_ADCS);
        checkFastCount(count);
        double[] value = {0.0, 0.0}, sValue = new double[2];
        for (int j = 0; j < count; j++) {
            readFastSample(j, sValue);
            value[0] += sValue[0];
            value[1] += sValue[1];
        }
        value[0] /= count;
        value[1] /= count;

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Checks the fast ADC sample count.
    **
    **  @param  count  The number of samples to read
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkFastCount(int count) throws REBException
    {
        if (count < 0 || count >= NUM_FADC_SAMPS) {
            throw new REBException("Invalid number of fast ADC samples");
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a pair of fast ADC samples.
    **
    **  @param  index  The index of the sample pair
    **
    **  @param  value  A two-element array to receive the sample values
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void readFastSample(int index, double[] value) throws REBException
    {
        int iValue = read(REG_FADC_SAMPS + index);
        value[0] = FADC_SCALE * (((iValue << 8) >> 20) & 0xfff);
        value[1] = FADC_SCALE * (((iValue << 20) >> 20) & 0xfff);
    }

}
