package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  CABAC access routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Cabac extends Asic {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_CABAC         = 0x500000,
        REG_CABAC_LOAD    = REG_CABAC,
        REG_CABAC_FETCH   = REG_CABAC + 1,
        REG_CABAC_RESET   = REG_CABAC + 1,
        REG_CABAC_RESET_0 = REG_CABAC + 2,
        REG_CABAC_READ    = REG_CABAC + 0x10,
        CABAC_OFF_TOP     = 0x10,
        CABAC_OFF_BOTTOM  = 0x20,
        CABAC_OFF_WRITE   = 0,
        CABAC_OFF_READ    = 0x100,
        NUM_CABAC_REGS    = 5,

        VALUE_MASK        = 0xffff,

        REG_CABAC_POWER   = 0xd00001,
        PWR_ENA_5V        = 0x01,
        PWR_ENA_3V        = 0x02,
        PWR_ENA_VEE       = 0x04,
        PWR_ENA_B5V       = 0x08,
        PWR_ENA_B3V       = 0x10;

    public final static int
        CABAC_FLD_OD_EXP0 = 0,
        CABAC_FLD_OD_EXP1 = 1,
        CABAC_FLD_OD_RDO0 = 2,
        CABAC_FLD_OD_RDO1 = 3,
        CABAC_FLD_PCLK0   = 4,
        CABAC_FLD_PCLK1   = 5,
        CABAC_FLD_PCLK2   = 6,
        CABAC_FLD_PCLK3   = 7,
        CABAC_FLD_SCLK0   = 8,
        CABAC_FLD_SCLK1   = 9,
        CABAC_FLD_SCLK2   = 10,
        CABAC_FLD_RG      = 11,
        CABAC_FLD_GD      = 12,
        CABAC_FLD_OG      = 13,
        CABAC_FLD_RD      = 14,
        CABAC_FLD_SPH     = 15,
        CABAC_FLD_SPL     = 16,
        CABAC_FLD_MUX0    = 17,
        CABAC_FLD_MUX1    = 18,
        CABAC_FLD_PULSE   = 19,
        CABAC_FLD_MUXE    = 20,

        CABAC_FLD_SPARE   = 15,
        CABAC_FLD_MUX     = 16,
        CABAC_FLD_MUX_OFF = 17,
        CABAC_FLD_TEST    = 18,
        CABAC_FLD_EXP_CLK = 20,
        CABAC_FLD_BIAS_HZ = 21;
        
   /**
    ***************************************************************************
    **
    **  Private constants.
    **
    ***************************************************************************
    */
    private final static int[]
        FIELD_LENGTHS  = {  8,   8,   8,   8,   8,   8,   8,   8,
                            8,   8,   8,   8,   8,   8,   8,   8,  
                            8,   4,   4,   1,   1},
        FIELD_STARTS   = {  0,   8,  16,  24,  32,  40,  48,  56,
                           64,  72,  80,  88,  96, 104, 112, 120,
                          128, 136, 140, 144, 145},
        FIELD_ADDRS    = { 8,  9, 10, 11,  0,  1,  2,  3,
                           4,  5,  6,  7, 13, 14, 12, 15,
                          16, 17, 20, 21, 18, 19};
    public final static int
        NUM_CABAC_FLDS_0  = 21,
        NUM_CABAC_FLDS    = 22;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Cabac()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public Cabac(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Gets the CABAC version.
    **
    **  @return  The version number: VERSION_0 for CABAC0; VERSION_1 for CABAC1
    **
    ***************************************************************************
    */
    public int getVersion()
    {
        try {
            return getVersion(OPTN_CABAC);
        }
        catch (REBException e) {
            return VERSION_UNSUPP;
        }
    }


   /**
    ***************************************************************************
    **
    **  Fetches the contents of a CABAC strip to the registers.
    **
    **  This transfers the contents of both (top and bottom) CABACs in the
    **  strip to the CABAC input registers in the FPGA.
    **
    **  @param  strip  The number of the strip to be fetched
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void fetch(int strip) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        checkStrip(strip);
        write(REG_CABAC_FETCH, strip);
    }


   /**
    ***************************************************************************
    **
    **  Loads a CABAC strip from the registers.
    **
    **  This transfers the contents of the CABAC output registers in the FPGA
    **  to both (top and bottom) CABACs in the strip.
    **
    **  @param  strip  The number of the strip to be loaded
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void load(int strip) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        checkStrip(strip);
        write(REG_CABAC_LOAD, strip);
    }


   /**
    ***************************************************************************
    **
    **  Resets a CABAC strip.
    **
    **  The contents of both (top and bottom) CABACs in the strip are reset
    **  to all zeroes.
    **
    **  @param  strip  The number of the strip to be reset
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void reset(int strip) throws REBException
    {
        int versn = getVersion(OPTN_CABAC);
        checkStrip(strip);
        write(versn == VERSION_0 ? REG_CABAC_RESET_0 : REG_CABAC_RESET, strip);
    }


   /**
    ***************************************************************************
    **
    **  Reads CABAC input registers in raw format.
    **
    **  The contents of the five input registers for the top or bottom CABAC
    **  are returned.
    **
    **  @param  side  The side to read: 0 = top; 1 = bottom
    **
    **  @return  The 5-element array of values read from the registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] readRaw(int side) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side) + CABAC_OFF_READ;
        int[] values = new int[NUM_CABAC_REGS];
        read(base, values);

        return values;
    }


   /**
    ***************************************************************************
    **
    **  Writes CABAC output registers in raw format.
    **
    **  The five output registers for the top or bottom CABAC are set to the
    **  supplied values.
    **
    **  @param  side    The side to write: 0 = top; 1 = bottom
    **
    **  @param  values  The 5-element array of values to be written
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeRaw(int side, int[] values) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side) + CABAC_OFF_WRITE;
        write(base, values);
    }


   /**
    ***************************************************************************
    **
    **  Copies CABAC input registers to the output ones.
    **
    **  The input registers for the top or bottom CABAC are copied to its
    **  output registers.
    **
    **  @param  side  The side number: 0 = top; 1 = bottom
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void copyRaw(int side) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int rbase = getBase(side) + CABAC_OFF_READ;
        int wbase = getBase(side) + CABAC_OFF_WRITE;
        int[] values = new int[NUM_CABAC_REGS];
        read(rbase, values);
        write(wbase, values);
    }


   /**
    ***************************************************************************
    **
    **  Sets one CABAC field value.
    **
    **  A single field is inserted into the correct location in the top or
    **  bottom CABAC's output registers.  Since the output registers are
    **  write-only, this can be done in a reasonable way only if the output
    **  registers were initially primed with the contents of the input
    **  registers, which were in turn obtained by fetching the CABAC's
    **  contents.
    **
    **  @param  side   The side number: 0 = top; 1 = bottom
    **
    **  @param  field  The field ID
    **
    **  @param  value  The value to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int side, int field, int value) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side);
        checkField(field);
        int beg = FIELD_STARTS[field];
        int end = beg + FIELD_LENGTHS[field];
        int mask = (1 << (end - beg)) - 1;
        int regBeg = beg >> 5;
        int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
        if (regEnd == regBeg) {
            updateReg(base, regBeg, mask << bitEnd, value << bitEnd);
        }
        else {
            updateReg(base, regBeg, mask >> (32 - bitEnd),
                      value >> (32 - bitEnd));
            updateReg(base, regEnd, mask << bitEnd, value << bitEnd);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets all CABAC field values.
    **
    **  The values for all fields of the top or bottom CABAC are set into its
    **  output registers, completely overwriting their contents.
    **
    **  @param  side   The side number: 0 = top; 1 = bottom
    **
    **  @param  value  The 21-element array of values to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int side, int[] value) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side) + CABAC_OFF_WRITE;
        int[] regs = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
        }
        for (int j = 0; j < NUM_CABAC_FLDS_0; j++) {
            int beg = FIELD_STARTS[j];
            int end = beg + FIELD_LENGTHS[j];
            int val = value[j] & ((1 << (end - beg)) - 1);
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
            }
        }
        write(base, regs);
    }


   /**
    ***************************************************************************
    **
    **  Sets several CABAC field values.
    **
    **  The specified fields are inserted into the correct locations in the
    **  top or bottom CABAC's output registers.  Since the output registers
    **  are write-only, this can be done in a reasonable way only if the
    **  output registers were initially primed with the contents of the input
    **  registers, which were in turn obtained by fetching the CABAC's
    **  contents.
    **
    **  @param  side   The side number: 0 = top; 1 = bottom
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields to set
    **
    **  @param  value  The array of values to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int side, int first, int count, int[] value)
        throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side);
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], masks = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
            masks[j] = 0;
        }
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int val = value[j] & mask;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
                masks[regBeg] |= mask << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
                masks[regBeg] |= mask >> (32 - bitEnd);
                masks[regEnd] |= mask << bitEnd;
            }
        }
        for (int j = 0; j < regs.length; j++) {
            if (masks[j] == 0) continue;
            if (masks[j] == -1) {
                write(base + CABAC_OFF_WRITE + j, regs[j]);
            }
            else {
                updateReg(base, j, masks[j], regs[j]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets one CABAC field value.
    **
    **  The value of the specified field is extracted from the top or bottom
    **  CABAC input registers.
    **
    **  @param  side   The side number: 0 = top; 1 = bottom
    **
    **  @param  field  The field ID
    **
    **  @return  The value of the field
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int get(int side, int field) throws REBException
    {
        return get(side, field, 1)[0];
    }


   /**
    ***************************************************************************
    **
    **  Gets all CABAC field values.
    **
    **  The values of all fields are extracted from the top or bottom CABAC
    **  input registers.
    **
    **  @param  side  The side number: 0 = top; 1 = bottom
    **
    **  @return  The 21-element array of field values
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] get(int side) throws REBException
    {
        return get(side, 0, NUM_CABAC_FLDS_0);
    }


   /**
    ***************************************************************************
    **
    **  Gets several CABAC field values.
    **
    **  The values of the specified fields are extracted from the top or
    **  bottom CABAC input registers.
    **
    **  @param  side   The side number: 0 = top; 1 = bottom
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields to get
    **
    **  @return  The array of field values
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] get(int side, int first, int count) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_0);
        int base = getBase(side) + CABAC_OFF_READ;
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], value = new int[count];
        read(base, regs);
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                value[j] = (regs[regBeg] >> bitEnd) & mask;
            }
            else {
                value[j] = ((regs[regBeg] << (32 - bitEnd))
                              | (regs[regEnd] >>> bitEnd)) & mask;
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a CABAC field.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (top and/or bottom) to write
    **
    **  @param  field   The CABAC field ID
    **
    **  @param  value   The value to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeField(int strips, int sides, int field, int value)
        throws REBException
    {
        writeField(strips, sides, field, 1, new int[]{value});
    }


   /**
    ***************************************************************************
    **
    **  Writes all CABAC fields.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (top and/or bottom) to write
    **
    **  @param  value   The array of values to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeField(int strips, int sides, int[] value)
        throws REBException
    {
        writeField(strips, sides, 0, NUM_CABAC_FLDS, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes several CABAC fields.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (top and/or bottom) to write
    **
    **  @param  first   The ID of the first field
    **
    **  @param  count   The number of fields to write
    **
    **  @param  value   The array of values to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeField(int strips, int sides, int first, int count,
                           int[] value) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_1);
        checkField(first, count);
        for (int j = 0; j < count; j++) {
            write(REG_CABAC_LOAD,
                  makeCommand(strips, sides, FIELD_ADDRS[first + j], 1,
                              value[j] & VALUE_MASK));
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a CABAC field.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @param  field   The CABAC field ID
    **
    **  @return  The array of read values, one per read strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] readField(int strips, int side, int field) throws REBException
    {
        int[][] value = readField(strips, side, field, 1);
        int[] value0 = new int[value.length];
        for (int j = 0; j < value.length; j++) {
            value0[j] = value[j][0];
        }

        return value0;
    }


   /**
    ***************************************************************************
    **
    **  Reads all CABAC fields.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @return  The array of read value arrays, one per read strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[][] readField(int strips, int side) throws REBException
    {
        return readField(strips, side, 0, NUM_CABAC_FLDS);
    }


   /**
    ***************************************************************************
    **
    **  Reads several CABAC fields.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @param  first   The ID of the first field
    **
    **  @param  count   The number of fields to read
    **
    **  @return  The array of read value arrays, one per read strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[][] readField(int strips, int side, int first, int count)
        throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_1);
        checkSide(side);
        checkField(first, count);
        int[][] value = new int[Integer.bitCount(strips)][count];
        for (int j = 0; j < count; j++) {
            write(REG_CABAC_LOAD,
                  makeCommand(strips, 1 << side, FIELD_ADDRS[first + j], 0, 0));
            for (int k = 0, l = 0; k < getNumStrips(); k++) {
                if (((strips >> k) & 1) != 0) {
                    value[l++][j] = read(REG_CABAC_READ + k);
                }
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of CABAC fields.
    **
    **  @return  The number of fields
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumFields() throws REBException
    {
        return getVersion(OPTN_CABAC) == VERSION_0 ? NUM_CABAC_FLDS_0
                                                   : NUM_CABAC_FLDS;
    }


   /**
    ***************************************************************************
    **
    **  Sets the power regulator enables.
    **
    **  @param  enables  The mask of enable bits
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setRegulator(int enables) throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_1);
        write(REG_CABAC_POWER, enables);
    }


   /**
    ***************************************************************************
    **
    **  Gets the power regulator enables.
    **
    **  @return  The mask of enable bits
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getRegulator() throws REBException
    {
        checkVersion(OPTN_CABAC, VERSION_1);
        return read(REG_CABAC_POWER);
    }


   /**
    ***************************************************************************
    **
    **  Gets the base register address for the CABAC.
    **
    **  @param  side  The side number
    **
    **  @return  The CABAC base address
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getBase(int side) throws REBException
    {
        checkSide(side);

        return REG_CABAC + (side == SIDE_BOTTOM ? CABAC_OFF_BOTTOM
                                                : CABAC_OFF_TOP);
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC strip number for validity.
    **
    **  @param  strip  The strip number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkStrip(int strip) throws REBException
    {
        if (strip < 0 || strip >= getNumStrips()) {
            throw new REBException("Invalid CABAC strip number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC side number for validity.
    **
    **  @param  cabac  The CABAC number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkSide(int side) throws REBException
    {
        if (side != SIDE_BOTTOM && side != SIDE_TOP) {
            throw new REBException("Invalid CABAC side number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC field ID for validity.
    **
    **  @param  field  The field ID
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkField(int field) throws REBException
    {
        if (field < 0 || field >= getNumFields()) {
            throw new REBException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of CABAC field IDs for validity.
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkField(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumFields()) {
            throw new REBException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Updates a CABAC set register using its corresponding get register.
    **
    **  @param  base   The CABAC base address
    **
    **  @param  reg    The register number
    **
    **  @param  mask   A mask specifying which bits of value are to be used
    **
    **  @param  value  The value to use
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void updateReg(int base, int reg, int mask, int value)
        throws REBException
    {
        write(base + CABAC_OFF_WRITE + reg,
              (read(base + CABAC_OFF_READ + reg) & ~mask) | (value & mask));
    }

}
