package org.lsst.ccs.drivers.reb.sim;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.Reader;
import java.util.Set;
import java.util.TreeSet;

/**
 * An address space represents a set of registers each mapped to a unique
 * address range.
 *
 * @author tonyj
 */
public class AddressSpace {

    private final Set<RegisterSet> addressSpace = new TreeSet<>((RegisterSet o1, RegisterSet o2) -> o1.baseAddress() - o2.baseAddress());

    /**
     * Add a register set to an address space
     *
     * @param rs Add new new register set representing one or more registers.
     */
    void add(RegisterSet rs) {
        addressSpace.add(rs);
    }

    private RegisterSet mapAddress(int address) {
        // Would be nice to use a binary search here, although performance
        // gain would likely be minimal
        for (RegisterSet rs : addressSpace) {
            if (rs.baseAddress() + rs.length() <= address) {
                continue;
            }
            if (rs.baseAddress() > address) {
                break;
            }
            return rs;
        }
        throw new SimulationException("Address 0x%08x not found in address space", address);
    }

    void readRegs(int address, int[] values, int offset, int count) {
        for (int i = 0; i < count; i++) {
            values[i + offset] = read(address + i);
        }
    }

    void writeRegs(int address, int[] values, int offset, int count) {
        for (int i = 0; i < count; i++) {
            write(address + i, values[i + offset]);
        }
    }

    public int read(int address) {
        return mapAddress(address).read(address);
    }

    public void write(int address, int value) {
        mapAddress(address).write(address, value);
    }

    /**
     * Dump out all registers in the specified address range.
     *
     * @param pw The PrintWriter to output to
     * @param start The start address
     * @param end The end address
     */
    public void hexDump(PrintStream pw, int start, int end) {
        // FIXME: This implementation is not bobust against
        // small blocks of registeres. See (commented out) test
        // case. Works for the cases we care about.
        for (RegisterSet rs : addressSpace) {
            if (rs.baseAddress() > end) {
                break;
            }
            if (rs.baseAddress() + rs.length() > start) {
                int[] buffer = new int[8];
                for (int row = Math.max(start,rs.baseAddress()); row < rs.baseAddress() + rs.length(); row += buffer.length) {
                    boolean allZero = true;
                    for (int address = row; address < end && address - row < buffer.length; address++) {
                        try {
                            final int value = read(address);
                            allZero &= (value == 0);
                            buffer[address - row] = value;
                        } catch (SimulationException x) {
                            buffer[address - row] = 0;
                        }
                    }
                    if (!allZero) {
                        pw.printf("%08x: ", row);
                        for (int value : buffer) {
                            pw.printf("%08x ", value);
                        }
                        pw.println();
                    }
                }
            }
        }
    }

    /**
     * Load a hexdump file into this address space.
     *
     * @param reader A reader for reading the file.
     * @throws IOException If an IOException occurs, or if the file has bad
     * content.
     */
    public void hexLoad(Reader reader) throws IOException {
        try (BufferedReader in = reader instanceof BufferedReader ? (BufferedReader) reader : new BufferedReader(reader)) {
            in.lines().forEach((String line) -> {
                String[] tokens = line.split(":?\\s+");
                if (tokens.length < 2) {
                    throw new SimulationException("Bad syntax for hex file at line\n\t" + line);
                }
                int address = Integer.parseInt(tokens[0], 16);
                for (int i = 1; i < tokens.length; i++) {
                    write(address++, Integer.parseUnsignedInt(tokens[i], 16));
                }
            });
        }
    }

}
