package org.lsst.ccs.drivers.reb;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 *  Program to test the Java image client interface routines
 *
 *  @author Owen Saxton
 */
public class TestImage implements ImageClient.Listener {

    /*
     *  Private fields
     */
    private static final String CUP = "\033[A", NL = "\n";
    private final ImageClient imc = new ImageClient();
    private final BaseSet reg = new BaseSet();
    private int rebId = -1, imgLength = -1, imgOffset = 0;
    private boolean listening, showTime, reuseImage = true;
    private Image image = new Image();


    /**
     *  Enumeration for on/off.
     */
    public enum OnOff {

        ON, OFF;

    }


    /**
     *  Enumeration for hardware type.
     */
    public enum HdwType {

        DAQ(RegClient.HDW_TYPE_DAQ),
        DAQ0(RegClient.HDW_TYPE_DAQ0),
        DAQ1(RegClient.HDW_TYPE_DAQ1),
        DAQ2(RegClient.HDW_TYPE_DAQ2),
        DAQ4(RegClient.HDW_TYPE_DAQ4),
        PCI0(RegClient.HDW_TYPE_PCI0),
        PCI1(RegClient.HDW_TYPE_PCI1);

        int value;

        HdwType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


    /**
     *  Connects to a REB.
     *
     *  @param  id  The REB ID
     *  @throws  REBException
     */
    @Command(description="Connect to a REB")
    public void connect(@Argument(description="REB ID") int id) throws REBException
    {
        imc.open(id);
        reg.open(id);
        rebId = id;
    }


    /**
     *  Connects to a REB connected via DAQ1/2/4.
     *
     *  @param  id     The REB ID
     *  @param  part   The partition name
     *  @throws  REBException
     */
    @Command(description="Connect to a REB")
    public void connect(@Argument(description="REB ID") int id,
                        @Argument(description="Partition name") String part) throws REBException
    {
        imc.open(id, part);
        try {
            reg.open(id, part);
        }
        catch (REBException e) {
            imc.close();
            throw e;
        }
        rebId = id;
    }


    /**
     *  Connects to a REB.
     *
     *  @param  type     The hardware type (DAQ, DAQ0, DAQ1, DAQ2. DAQ4, PCI0, PCI1)
     *  @param  id       The REB ID
     *  @param  ifcName  The partition, network interface, or PCI device name
     *  @throws  REBException
     */
    @Command(description="Connect to a REB")
    public void connect(@Argument(description="Hardware type") HdwType type,
                        @Argument(description="REB ID") int id,
                        @Argument(description="Network interface name") String ifcName) throws REBException
    {
        imc.open(type.getValue(), id, ifcName);
        try {
            reg.open(type.getValue(), id, ifcName);
        }
        catch (REBException e) {
            imc.close();
            throw e;
        }
        rebId = id;
    }


    /**
     *  Disconnects from a REB.
     *
     *  @throws  REBException
     */
    @Command(description="Disconnect from a REB")
    public void disconnect() throws REBException
    {
        rebId = -1;
        try {
            reg.close();
        }
        catch (REBException e) {
            imc.close();
            throw e;
        }
        imc.close();
    }


    /**
     *  Turns image data buffer re-use on or off.
     *
     *  @param  action  The buffer re-use action: ON or OFF
     */
    @Command(description="Enable or disable image buffer re-use")
    public void reuse(@Argument(description="Action to take") OnOff action)
    {
        reuseImage = (action == OnOff.ON);
    }


    /**
     *  Turns image acquisition time display on or off.
     *
     *  @param  action  The display action: ON or OFF
     */
    @Command(description="Enable or disable time display")
    public void time(@Argument(description="Action to take") OnOff action)
    {
        showTime = (action == OnOff.ON);
    }


    /**
     *  Turns the listener on or off.
     *
     *  @param  action  The listener action: ON or OFF
     *  @return  "Cursor up" string if success; error message otherwise
     *  @throws  REBException
     */
    @Command(description="Enable or disable the listener")
    public String listen(@Argument(description="Action to take") OnOff action) throws REBException
    {
        String result = checkConnect();
        if (result != null) return result;
        if (action.ordinal() == 0) {
            listening = true;
            imc.setListener(this, reuseImage ? image : null);
        }
        else {
            listening = false;
            imc.clearListener();
        }
        return CUP;
    }


    /**
     *  Sets the internal image display region.
     *
     *  @param  length  The length of the display region
     */
    @Command(description="Set the image display region")
    public void region(@Argument(description="The region length (pixels)") int length)
    {
        imgLength = length;
    }


    /**
     *  Sets the internal image display region.
     *
     *  @param  length  The length of the display region
     *  @param  offset  The offset to the start of the region
     */
    @Command(description="Set the image display region")
    public void region(@Argument(description="The region length (pixels)") int length,
                       @Argument(description="The offset to the region (pixels)") int offset)
    {
        imgLength = length;
        imgOffset = offset;
    }


    /**
     *  Displays parameters.
     *
     *  @return  The result string
     */
    @Command(description="Display parameters")
    public String show()
    {
        return String.format(  "REB ID       = %-8s  Image length = %s" +
                             "\nImage offset = %-8s  Listener on  = %s" +
                             "\nShow time    = %-8s  Reuse image  = %s",
                             rebId, imgLength, imgOffset, listening, showTime, reuseImage);
    }


    /**
     *  Waits for an image to arrive.
     *
     *  @return  The result string
     *  @throws  REBException
     */
    @Command(name="wait", description="Wait for an image to arrive")
    public String awaitImage() throws REBException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        image = imc.awaitImage(reuseImage ? image : null);
        if (showTime) {
            long cTime = reg.getTime();
            long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
            result = "Elapsed time = " + (cTime - tTime) + " msec";
        }
        String text = formatMetadata(image, "Got metadata", false);
        return (result == null) ? text : result + NL + text;
    }


    /**
     *  Reads an image.
     *
     *  @return  The result string
     *  @throws  REBException
     */
    @Command(description="Read an image")
    public String read() throws REBException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        long start = System.currentTimeMillis();
        boolean imgOkay = imc.readImage(image);
        if (showTime) {
            result = "Elapsed time = " + (System.currentTimeMillis() - start) + " msec";
        }
        if (!imgOkay) {
            String text = "Error reading image data";
            result = (result == null) ? text : result + NL + text;
        }
        return (result == null) ? CUP : result;
    }


    /**
     *  Interrupts an image wait.
     *
     *  @throws  REBException
     */
    @Command(description="Interrupt an image wait")
    public void interrupt() throws REBException
    {
        imc.interrupt();
    }


    /**
     *  Displays image data.
     *
     *  @param  length  The number of pixels to display
     *  @return  The result string
     */
    @Command(description="Display image data")
    public String display(@Argument(description="The display region length (pixels)") int length)
    {
        return display(0, 0, length);
    }


    /**
     *  Displays image data.
     *
     *  @param  length  The number of pixels to display
     *  @param  offset  The offset to the first pixel
     *  @return  The result string
     */
    @Command(description="Display image data")
    public String display(@Argument(description="The display region length (pixels)") int length,
                          @Argument(description="The offset to the display region (pixels)") int offset)
    {
        return display(0, offset, length);
    }


    /**
     *  Displays image data.
     *
     *  @param  ccd     The CCD number
     *  @param  length  The number of pixels to display
     *  @param  offset  The offset to the first pixel
     *  @return  The result string
     */
    @Command(description="Display image data")
    public String display(@Argument(description="The CCD number") int ccd,
                          @Argument(description="The display region length (pixels)") int length,
                          @Argument(description="The offset to the display region (pixels)") int offset)
    {
        String text = formatImage(image, ccd, offset, length);
        return (text == null) ? CUP : text;
    }


    /**
     *  Displays image metadata.
     *
     *  @return  The result string
     */
    @Command(description="Display image metadata")
    public String metadata()
    {
        return image.getMetadata().toString();
    }


    /**
     *  Resets the front end system.
     *
     *  @throws  REBException
     */
    @Command(description="Reset the front end")
    public void reset() throws REBException
    {
        imc.resetFrontEnd();
    }


    /**
     *  Handles each received image.
     *
     *  @param  img   The received image
     */
    @Override
    public void processImage(Image img)
    {
        image = img;
        if (showTime) {
            try {
                long cTime = reg.getTime();
                long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
                System.out.println("Elapsed time = " + (cTime - tTime) + " msec");
            }
            catch (REBException e) {            
            }
        }
        System.out.println(formatMetadata(image, "Got image", true));
        if (image.getData() != null) {
            String text = formatImage(image, 0, imgOffset, imgLength);
            if (text != null) {
                System.out.println(text);
            }
        }
    }


    /**
     *  Formats the metadata for an image.
     *
     *  @param  img        The image whose metadata is to be displayed
     *  @param  caption    A caption to be displayed before the metadata
     *  @param  showValid  If true, display whether or not the image contains valid data
     *  @return  A string containing the formatted metadata
     */
    private String formatMetadata(Image img, String caption, boolean showValid)
    {
        StringBuilder blanks = new StringBuilder(caption.length());
        for (int j = 0; j < caption.length(); j++) {
            blanks.append(" ");
        }
        return String.format("%s: name = %s, tstamp = %016x" + NL
                               + "%s  length = %s, partition = %s, address = %s, status = %s",
                             caption, image.getName(), image.getTimestamp(),
                             blanks, image.getLength(), image.getPartition(), image.getAddress(), image.getStatus());
    }


    /**
     *  Formats the contents of an image.
     *
     *  @param  img     The image to be displayed
     *  @param  ccd     The CCD number (ignored for interleaved data)
     *  @param  offset  The offset to the first pixel to display
     *  @param  count   The number of pixels to display.
     *  @return  The formatted image contents
     */
    private String formatImage(Image img, int ccd, int offset, int count)
    {
        if (count < 0) {
            return null;
        }
        ByteBuffer buff = img.isInterleaved() ? img.getData() : img.getData(ccd);
        if (buff == null || img.getStatus() != Image.STATUS_GOOD) {
            return "Image contains no valid data for CCD " + ccd;
        }

        offset = (offset < 0) ? 0 : offset;
        int iCount = img.getLength();
        if (count + offset > iCount) {
            count = iCount - offset;
        }
        if (count <= 0) {
            return null;
        }

        StringBuilder text = new StringBuilder();
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        for (int j = 0; j < count; j++) {
            if ((j & 0x03) == 0) {
                if (j != 0) {
                    text.append(NL);
                }
                text.append(String.format("%08x:", offset + j));
            }
            text.append(String.format(" %08x", buff.getInt()));
        }

        return text.toString();
    }


    /**
     *  Checks that a REB is connected.
     */
    private String checkConnect()
    {
        return (imc != null) ? null : "No REB connected";
    }


    /**
     *  Checks that the listener is not active.
     */
    private String checkListen()
    {
        return !listening ? null : "Listener is active";
    }

}
