package org.lsst.ccs.drivers.reb.sim;

/**
 * Register set is a set of classes for handling one or more registers mapped to
 * an address space.
 *
 * @author tonyj
 */
public interface RegisterSet {

    /**
     * The base address of this set of registers.
     *
     * @return The base address of this set of registers.
     */
    int baseAddress();

    /**
     * The length of this register set.
     *
     * @return The length (in 32-byte words) of this register set.
     */
    int length();

    /**
     * Write a single 32 bit value to this register set
     *
     * @param address The absolute address to write to
     * @param value The value to be written
     */
    void write(int address, int value);

    /**
     * Read a single 32 bit value from this register set
     *
     * @param address The absolute address to read from
     * @return The value read.
     */
    int read(int address);

    /**
     * SimpleRegisterSet implements the RegisterSet interface with a fixed
     * baseAddress and length. It checks for valid address arguments to read and
     * write and converts the addresses passed to readSimple and writeSimple to
     * be relative to the baseAddress.
     */
    static abstract class SimpleRegisterSet implements RegisterSet {

        private final int baseAddress;
        private final int length;

        /**
         * Create a SimpleRegisterSet
         *
         * @param address The base address for this register set
         * @param length The length (in 32bit words) of this register set
         */
        SimpleRegisterSet(int address, int length) {
            this.baseAddress = address;
            this.length = length;
        }

        @Override
        public int read(int address) {
            return readSimple(checkAddress(address));
        }

        @Override
        public void write(int address, int value) {
            writeSimple(checkAddress(address), value);
        }

        @Override
        public int length() {
            return length;
        }

        @Override
        public int baseAddress() {
            return baseAddress;
        }

        private int checkAddress(int address) {
            if (address < baseAddress | address >= baseAddress + length) {
                throw new SimulationException("Invalid address 0x%08x", address);
            }
            return address - baseAddress;
        }

        /**
         * Read a single address.
         *
         * @param relativeAddress The address, relative to the baseAddress of
         * this register set.
         * @return The value read.
         */
        abstract int readSimple(int relativeAddress);

        /**
         * Write a single address.
         *
         * @param relativeAddress The address, relative to the baseAddress of
         * this register set
         * @param value The value to be written
         */
        abstract void writeSimple(int relativeAddress, int value);
    }

    /**
     * A set of registers mapped to an array of longs
     */
    static class LongArrayRegisterSet extends SimpleLongRegisterSet {

        private final long[] data;

        /**
         * Create a set of registers backed by a long array
         * @param address The base address
         * @param data The long array
         */
        LongArrayRegisterSet(int address, long[] data) {
            super(address, data.length);
            this.data = data;
        }

        @Override
        long readLong(int index) {
            return data[index];
        }

        @Override
        void writeLong(int index, long value) {
            data[index] = value;
        }
    }

    /** 
     * A sequence of longs mapped to a register set. Each long value has two 
     * addresses, the first (lower) address represents the least significant 32 bits,
     * and the second represents the most significant 32 bits.
     */
    static abstract class SimpleLongRegisterSet extends SimpleRegisterSet {

        private static final long HIGH32BITS = 0xffffffff00000000l;
        private static final long LOW32BITS = 0xffffffffl;

        SimpleLongRegisterSet(int address, int length) {
            super(address, 2 * length);
        }

        @Override
        int readSimple(int address) {
            long data = readLong(address / 2);
            if (address % 2 == 0) {
                return (int) (data & 0xffffffff);
            } else {
                return (int) (data >> 32);
            }
        }

        @Override
        void writeSimple(int address, int value) {
            long longValue = readLong(address / 2);
            if (address % 2 == 0) {
                longValue = (longValue & HIGH32BITS) | (value & LOW32BITS);
            } else {
                longValue = (longValue & LOW32BITS) | (((long) value) << 32);
            }
            writeLong(address / 2, longValue);
        }

        /**
         * Read a single long value
         * @param index The index within the set of longs
         * @return The long value at the given index
         */
        abstract long readLong(int index);

        /**
         * Write a single long value
         * @param index The index within the set of longs
         * @param longValue The value to write at the given index
         */
        abstract void writeLong(int index, long longValue);

    }
    
    /**
     * Implementation of a register consisting of a single register.
     */
    static abstract class Register implements RegisterSet {

        private final int baseAddress;

        /**
         * Create a single register.
         *
         * @param baseAddress The address of the register
         */
        Register(int baseAddress) {
            this.baseAddress = baseAddress;
        }

        @Override
        public int read(int address) {
            checkAddress(address);
            return read();
        }

        @Override
        public void write(int address, int value) {
            checkAddress(address);
            write(value);
        }

        @Override
        public int length() {
            return 1;
        }

        @Override
        public int baseAddress() {
            return baseAddress;
        }

        /**
         * Write the value of the register
         *
         * @param value
         */
        public abstract void write(int value);

        /**
         * Read the value of the register
         *
         * @return
         */
        public abstract int read();

        private void checkAddress(int address) {
            if (address != baseAddress) {
                throw new SimulationException("Invalid address 0x%08x", address);
            }
        }

    }

    /**
     * A read only single register.
     */
    public static abstract class ReadOnlyRegister extends Register {
        
        /**
         * Create a read-only register
         * @param baseAddress The address of the register
         */
        ReadOnlyRegister(int baseAddress) {
            super(baseAddress);
        }

        @Override
        public void write(int value) {
            throw new SimulationException("Attempt to write to read-only register");
        }

    }

    /**
     * A read only single register.
     */
    public static abstract class WriteOnlyRegister extends Register {
        
        /**
         * Create a read-only register
         * @param baseAddress The address of the register
         */
        WriteOnlyRegister(int baseAddress) {
            super(baseAddress);
        }

        @Override
        public int read() {
            throw new SimulationException("Attempt to read from write-only register");
        }

    }    
    /**
     * A read only single register with a constant value
     */
    static class ConstantRegister extends ReadOnlyRegister {

        private final int value;

        /**
         * Create a read-only constant valued register
         * @param address The address of the register
         * @param value The value of the register.
         */
        public ConstantRegister(int address, int value) {
            super(address);
            this.value = value;
        }

        @Override
        public int read() {
            return value;
        }

    }
    /**
     * A register set wrapping an int array.
     */
    static class ArrayRegisterSet extends SimpleRegisterSet {

        private final int[] data;
        /**
         * Create an ArrayRegisterSet
         * @param address The base address of the register set
         * @param data The array
         */
        ArrayRegisterSet(int address, int[] data) {
            super(address, data.length);
            this.data = data;
        }

        @Override
        int readSimple(int address) {
            return data[address];
        }

        @Override
        void writeSimple(int address, int value) {
            data[address] = value;
        }
    }
    
    static abstract class ReadOnlyLongRegister extends SimpleLongRegisterSet {
        ReadOnlyLongRegister(int address) {
            super(address,1);
        }

        @Override
        long readLong(int index) {
            return read();
        }

        @Override
        void writeLong(int index, long longValue) {
            throw new SimulationException("Attempt to write to read-only register");
        }

        public abstract long read();
    }
}
