package org.lsst.ccs.drivers.reb.sim;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.Image;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.sim.SequencerSimulation.StateListener;

/**
 * A simulation of the DAQv1 image client with a REB3
 *
 * @author tonyj
 */
public class ImageClientDaq1Simulation implements ImageClient.Impl {

    private final HandleAndIdManager manager;
    private REB3Simulation reb3;
    private final DAQSimulation daq;
    private final AutoCloseableReentrantLock stateLock = new AutoCloseableReentrantLock();
    private final Condition imageDelivered = stateLock.newCondition();
    private final PixelCounter pixelCounter = new PixelCounter();
    private int schema;
    private int version;
    private int id;
    private long tag;
    private int length;

    private final StateListener stateListener = (SequencerSimulation.State oldState, SequencerSimulation.State newState) -> {
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            if (newState == SequencerSimulation.State.RUNNING) {
                tag = reb3.getTriggerTime(BaseSet.RSET_SEQUENCER);
                pixelCounter.reset();
            }
            if (oldState == SequencerSimulation.State.RUNNING && newState == SequencerSimulation.State.STOPPED) {
                length = pixelCounter.getPixels() * 16 * 4;
                if (length > 0) {
                   imageDelivered.signalAll();
                }
            }
        }
    };
    
    ImageClientDaq1Simulation(HandleAndIdManager manager, DAQSimulation daq) {
        this.manager = manager;
        this.daq = daq;
    }

    @Override
    public void newImageClient(int id, String ifc) {
        // Note: The IFC is not actually used in this (or any?) implementation.
        // Note: My guess is that newImageClient is actually only called once 
        // for any instance of ImageClient.Impl, otherwise the other methods
        // of this class do not really make sense.
        AddressSpace rs = manager.getAddressSpaceForId(id);
        if (rs instanceof REB3Simulation) {
            if (reb3 != null) {
                throw new SimulationException("Unsupported repeat call to newImageClient");
            }
            reb3 = (REB3Simulation) rs;
            reb3.getSequencer().addStateListener(stateListener);
            reb3.getSequencer().addWaveformListener(pixelCounter);
            schema = reb3.read(BaseSet.REG_SCHEMA);
            version = reb3.read(BaseSet.REG_VERSION);
            this.id = reb3.read(BaseSet.REG_ID);
        } else {
            throw new SimulationException("Unsupported implementation of AddressSpace returned");
        }
    }

    @Override
    public void deleteImageClient() {
        if (reb3 != null) {
            reb3.getSequencer().removeStateListener(stateListener);
            reb3.getSequencer().removeWaveformListener(pixelCounter);
        }
    }

    @Override
    public Image waitForImage(Image image) {
        SimulatedImage result = image instanceof SimulatedImage ? (SimulatedImage) image : new SimulatedImage(this);
        result.setSchema(schema);
        result.setVersion(version);
        result.setAddress(id);
        result.setRebType(daq.getRebType());
        // Note: It is not clear if we should return immediately if an image has been delivered
        // already. For now we always wait.
        try (AutoCloseableReentrantLock lock = stateLock.open()) {
            imageDelivered.await();
            result.setLength(length * result.getNumCcds());
            result.setTag(tag);
            return result;
        } catch (InterruptedException ex) {
            throw new SimulationException("Interrupted while waiting for image", ex);
        }
    }

    @Override
    public boolean getImage(Image image) {
        if (image instanceof SimulatedImage) {
            SimulatedImage simImage = (SimulatedImage) image;
            //When the image is produced we fetch the values of
            //the registers in which we stored the image metadata.
            //Is this the right place for it?
            int[] registers = daq.getRegisters();
            int[] metadata = null;
            if ( registers != null ) {            
                metadata = new int[registers.length];
                for (int i = 0; i < registers.length; i++) {
                    reb3.readRegs(registers[i], metadata, i, 1);
                }
            }
            simImage.setRegisters(metadata);
            simImage.setName(daq.getImageName());
            simImage.setData(daq.getDataProvider().getData(image.getMetadata()));
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void reset() {
        // This should reset the reb3.
    }

    @Override
    public void deleteImageMetadataRef(Image image) {
        // Nothing to be done.
    }

}
