/**
 * Allows communication with a Parker Hannifin ACR-class motor
 * controller over Ethernet.
 * <p>
 * Parker's ACR-class controllers can communicate over old-fashioned COM ports, USB or Ethernet.
 * This driver package uses Ethernet, so you must either be prepared to use the factory-standard
 * IP address of 192.168.10.40 or set up your own IP address using one of the other methods
 * of communication. For an ACR9000 we had trouble getting the COM port to work, apparently
 * due to cabling problems. Using USB in combination with Parker's ACR View program
 * worked well.
 * <p>
 * Once you have the IP address set you can connect to the controller
 * on two basic TCP ports. Port 5002 just takes straight ASCII commands; you can connect to it
 * via telnet and just type away. This package uses port 5006, the combined binary/ASCII
 * command port. Binary commands and their replies are of known lengths and start with a
 * marker byte containing either 0x00 or 0x1d. Binary commands let you fetch parameters
 * and program variables in a binary form that needs little conversion. A transmission
 * starting with a byte other than one of the marker values is treated as straight ASCII
 * save that it must be zero-padded to a multiple of four bytes. Replies to ASCII commands
 * are of variable length and we rely on a timeout to tell us when the reply has ended.
 * <p>
 * The controller has hundreds of different parameters that can be classified by data type
 * and by which controller object they belong to (if any). The data types are:
 * <ul>
 * <li>Long - a 32-bit integer. Depending on the parameter it may be signed or unsigned,
 *     though for AcroBasic variables it's always signed. This driver uses Java longs
 *     to hold these values, sign-extending them if appropriate.
 * </li>
 * <li>FP32 - a.k.a Single, a 32-bit floating point value in whatever format the
 *     the controller uses internal, which varies among models. A binary fetch
 *     of an FP32 value returns it converted to IEEE single precision format.
 *     The driver uses Java doubles to hold these values.
 * </li>
 * <li>FP64 - a.k.a. Double, a 64-bit floating point value in some model-dependent format.
 * A binary fetch of an FP64 value returns it converted to IEEE single precision.
 * Only the User parameters (see below) and AcroBasic program variables
 * may have this type.
 * </li>
 * </ul>
 * In addition to parameters there are bit flags, which are also divided into groups
 * based on the controller object they control or describe. This driver
 * represents a set bit as {@code true} and a clear bit as {@code false}.
 * <p>
 * These are the categories into which parameters and flags are divided:
 * <ul>
 * <li>User - set aside for the application to which to controller will be put.</li>
 * <li>Program - related to one of the AcroBasic programs the controller can
 *     store and run.
 * </li>
 * <li>Master - related to one of the axis-control masters.</li>
 * <li>Axis -related to one of the motion axes (motor+encoder+driver).</li>
 * <li>Encoder - related to one of the motor position encoders.</li>
 * <li>Connection - one of the controller's means of communication with the user.</li>
 * <li>System - general controller operations not tied to one of the object types above.</li>
 * </ul>
 * This package contains enumeration classes giving names to subsets of the
 * controller objects: 32 user parameters, eight programs, eight masters,
 * eight axes, eight encoders and eight connections. For example, the class
 * {@code AxisName} contains the names for the axes. Some of the objects named by these
 * classes may not be implemented in your controller.
 * <p>
 * There are also enumeration classes for various parameters and bits where each
 * class handles one combination of data type and controller object type. There
 * are hundreds of parameters and bits; what we've selected here are the ones which
 * turned out to be useful for the camera shutter but others can be added. Each
 * enumerator will in general have two instance methods: {@code index()} which gives
 * the numerical index the controller uses for the parameter/bit and {@code reference()}
 * which gives the string used to refer to the parameter/bit in AcroBasic code.
 * These methods will require an object name as argument for parameters/bits
 * that are tied to specific controller objects. For example
 * {@code AxisSingle.JOG_VELOCITY_SETTING.index(AxisName.AXIS0)} returns 12348 while
 * {@code AxisSingle.JOG_VELOCITY_SETTING.reference(AxisName.AXIS0)} returns "P12348".
 * <p>
 * There are similar enumerations, such as {@code LocalDouble} for a certain number of
 * AcroBasic variables local to controller programs, made possible by the rigid naming scheme
 * employed by the AcroBasic interpreter. See the description of class {@code AcrComm}
 * for a description of how variables are named.
 * <p>
 * A set of methods in {@code AcrComm} named simply {@code get()} or {@code set()}
 * allow you to get and set (if not read-only) all the enumerated parameters, bits
 * and local program variables. Since each enumeration belongs to a specific class,
 * method overloading allows the Java compiler to select the implementation that uses the
 * correct binary command and data encoding. If applicable the method takes an axis,
 * master, program, etc., name in addition to the name of the parameter/bit/variable.
 */
package org.lsst.ccs.drivers.parker;
