package org.lsst.ccs.drivers.lighthouse;

import org.lsst.ccs.drivers.commons.DriverException;

import java.time.Duration;
import java.time.Instant;

import java.util.Set;

/**
 * Provides low-level hardware operations.
 * @author tether
 */
public interface Hardware {
    /**
     * Reads the model designation registers.
     * @return The register values made into a String.
     * @throws DriverException if communication problems arise.
     */
    String readDeviceModel() throws DriverException;

    /**
     * Reads the Modbus map version register.
     * @return The version number times 100. v1.44 would be represented as 144.
     * @throws DriverException if communication problems arise.
     */
    int readDeviceMapVersion() throws DriverException;

    /**
     * Reads the device status register.
     * @return A set with one element per bit set in the register.
     * @throws DriverException if communication problems arise.
     */
    Set<DeviceFlag> readDeviceFlags() throws DriverException;

    /**
     * Writes to the device command register.
     * @param cmd A valid device command.
     * @throws DriverException if communication problems arise.
     */
    void writeDeviceCommand(DeviceCommand cmd) throws DriverException;

    /**
     * Reads the record count register.
     * @return The number of records currently held in the device's memory.
     * @throws DriverException if communication problems arise.
     */
    int readRecordCount() throws DriverException;

    /**
     * Writes to the record index register, making the data registers show the contents
     * of a particular record in its memory.
     * @param index The zero-based record index. 0 .. N - 1, or -1 for the latest record.
     * @throws DriverException if communication problems arise.
     */
    void writeRecordIndex(int index) throws DriverException;

    /**
     * Reads the record timestamp register.
     * @return The time at which data collection started for the current record.
     * @throws DriverException if communication problems arise.
     */
    Instant readRecordStartTime() throws DriverException;

    /**
     * Reads the record sample time register.
     * @return How long data collection took for the current record.
     * @throws DriverException if communication problems arise.
     */
    Duration readRecordDuration() throws DriverException;

    /**
     * Reads the registers for location number and location name.
     * @return The name of the device location for the current record.
     * @throws DriverException if communication problems arise.
     */
    String readRecordLocation() throws DriverException;

    /**
     * Reads the device status registers of the current record.
     * @return The set of RecordFlag corresponding to the status mask.
     * See {@link RecordFlag#flagsFromMask(int) }.
     * @throws DriverException if communication problems arise.
     */
    Set<RecordFlag> readRecordFlags() throws DriverException;

    /**
     * Reads the data type registers for a data channel of the current record.
     * @param chan The channel.
     * @return The channel type.
     * @throws DriverException if communication problems arise or
     * if the channel type label read from the device isn't what's specified by {@code chan}.
     */
    ChannelType readChannelType(DataChannel chan) throws DriverException;

    /**
     * Reads the units registers for a data channel of the current record.
     * @param chan The channel.
     * @return The units.
     * @throws DriverException if communication problems arise or
     * if the units label read from the device isn't what's specified by {@code chan}.
     */
    ChannelUnits readChannelUnits(DataChannel chan) throws DriverException;

    /**
     * Reads the value registers for a data channel of the current record. Assumes that
     * the registers hold a single-precision IEEE float for an analog channel or an int for others.
     * @param chan The channel.
     * @return The value, converted to double.
     * @throws DriverException if communication problems arise or
     * if the channel isn't enabled on the device.
     */
    double readChannelValue(DataChannel chan) throws DriverException;
}
