package org.lsst.ccs.drivers.lighthouse;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * The kinds of things Lighthouse data channel might measure. By no means exhaustive.
 * @author tether
 */
public enum ChannelType {
    /**
     * A particle count. The "type" as reported by the device is the particle size in microns.
     */
    COUNTER("counter", "0.2", "0.3", "0.5", "1.0", "2.5", "5.0", "10.0"),
    
    /**
     * A temperature.
     */
    TEMPERATURE("temperature", "TEMP"),
    
    /**
     * A relative humidity.
     */
    RELATIVE_HUMIDITY("relhumidity", "RH");
    
    private final String fileText;
    private final String[] deviceLabels;
    
    ChannelType(String text, String... deviceLabels) {
        this.fileText = text;
        this.deviceLabels = deviceLabels;
    }
    
    /**
     * Gets the text form of the channel type.
     * @return The text.
     */
    @Override
    public String toString() {return fileText;}
    
    private static final Map<String, ChannelType> labelMap;
    
    static {
        final Map<String, ChannelType> lmap = new HashMap<>();
        for (ChannelType ct : ChannelType.values()) {
            for (String lb : ct.deviceLabels) {
                lmap.put(lb, ct);
            }
        }
        labelMap = Collections.unmodifiableMap(lmap);
    }
    
    /**
     * Find the measurement type corresponding to a device type-label.
     * @param label The data type label used by the counter.
     * @return The corresponding ChannelType enumerator.
     */
    public static ChannelType fromTypeLabel(String label) {return labelMap.get(label);}
}
